import { datatype, lorem, random } from 'faker';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import type { NonNullableObj } from 'tachyon-type-library';
import type { EventTrackerContext, TwitchTracking } from '../EventTrackerRoot';
import { eventTrackerContext } from '../EventTrackerRoot';
import type { PageviewEventProperties } from '../event-data';
import { EventTrackerVODType, EventType } from '../event-data';
import { Pageview } from '.';

describe(Pageview, () => {
  function contextWithTracking(
    twitchTracking: TwitchTracking = {},
  ): EventTrackerContext {
    return {
      interactionMedium: '',
      location: undefined,
      onEvent: jest.fn(),
      twitchTracking,
    };
  }

  const setup = createMountWrapperFactory(Pageview, undefined, {
    wrappingContexts: () => [[eventTrackerContext, contextWithTracking()]],
  });

  const emptyEvent: PageviewEventProperties = {
    channel: undefined,
    channel_id: undefined,
    client_offline: undefined,
    content: undefined,
    email_id: undefined,
    game: undefined,
    is_live: undefined,
    location: undefined,
    medium: undefined,
    vod_id: undefined,
    vod_type: undefined,
  };

  it('reports basic pageview, defaulting all values to null', () => {
    const { contexts } = setup();

    expect(contexts.get(eventTrackerContext).onEvent).toHaveBeenCalledWith({
      event: EventType.Pageview,
      properties: expect.objectContaining({
        ...emptyEvent,
      }),
    });
  });

  it('reports full pageview', () => {
    const eventProperties: NonNullableObj<PageviewEventProperties> = {
      channel: lorem.word(),
      channel_id: datatype.number(),
      client_offline: true,
      content: lorem.word(),
      email_id: datatype.uuid(),
      game: lorem.words(4),
      is_live: datatype.boolean(),
      location: random.locale(),
      medium: lorem.word(),
      vod_id: datatype.uuid(),
      vod_type: EventTrackerVODType.Archive,
    };

    const { contexts } = setup(
      {
        ...eventProperties,
        channelID: eventProperties.channel_id!.toString(),
        clientOffline: true,
        isLive: eventProperties.is_live,
        vodID: eventProperties.vod_id,
        vodType: eventProperties.vod_type,
      },
      {
        wrappingContexts: [
          [
            eventTrackerContext,
            contextWithTracking({
              content: eventProperties.content!,
              emailID: eventProperties.email_id!,
              medium: eventProperties.medium!,
            }),
          ],
        ],
      },
    );

    expect(contexts.get(eventTrackerContext).onEvent).toHaveBeenCalledWith({
      event: EventType.Pageview,
      properties: expect.objectContaining({
        ...eventProperties,
        game: eventProperties.game!.toLowerCase(),
      }),
    });
  });

  it('reports full falsy-valued pageview without clobbering', () => {
    const eventProperties: NonNullableObj<PageviewEventProperties> = {
      channel: '',
      channel_id: 0,
      client_offline: undefined as any,
      content: '',
      email_id: '',
      game: '',
      is_live: false,
      location: '',
      medium: '',
      vod_id: '0',
      vod_type: EventTrackerVODType.Archive,
    };

    const { contexts } = setup(
      {
        ...eventProperties,
        channelID: eventProperties.channel_id!.toString(),
        isLive: eventProperties.is_live,
        vodID: eventProperties.vod_id,
        vodType: eventProperties.vod_type,
      },
      {
        wrappingContexts: [
          [
            eventTrackerContext,
            contextWithTracking({
              content: eventProperties.content!,
              emailID: eventProperties.email_id!,
              medium: eventProperties.medium!,
            }),
          ],
        ],
      },
    );

    expect(contexts.get(eventTrackerContext).onEvent).toHaveBeenCalledWith({
      event: EventType.Pageview,
      properties: expect.objectContaining({
        ...eventProperties,
      }),
    });
  });

  it('reports pageview with twitch tracking roundtrip', () => {
    const content = lorem.word();
    const medium = lorem.word();
    const emailID = datatype.uuid();

    const { contexts } = setup(undefined, {
      wrappingContexts: [
        [
          eventTrackerContext,
          contextWithTracking({ content, emailID, medium }),
        ],
      ],
    });

    expect(contexts.get(eventTrackerContext).onEvent).toHaveBeenCalledWith({
      event: EventType.Pageview,
      properties: expect.objectContaining({
        ...emptyEvent,
        content,
        email_id: emailID,
        medium,
      }),
    });
  });

  it('reports pageview with location from Pageview object', () => {
    const location = random.locale();
    const { contexts } = setup({ location });

    expect(contexts.get(eventTrackerContext).onEvent).toHaveBeenCalledWith({
      event: EventType.Pageview,
      properties: expect.objectContaining({
        ...emptyEvent,
        location,
      }),
    });
  });

  it('does not report twice due to a re-render', () => {
    const { contexts, wrapper } = setup({ location: random.locale() });
    expect(contexts.get(eventTrackerContext).onEvent).toHaveBeenCalledTimes(1);

    wrapper.update();
    expect(contexts.get(eventTrackerContext).onEvent).toHaveBeenCalledTimes(1);
  });

  it('reports when location changes', () => {
    const { contexts, wrapper } = setup();
    expect(contexts.get(eventTrackerContext).onEvent).toHaveBeenCalledTimes(1);

    wrapper.setProps({ location: random.locale() });
    expect(contexts.get(eventTrackerContext).onEvent).toHaveBeenCalledTimes(2);
  });
});
