import type { ComponentType, FC } from 'react';
import { getComponentName } from 'tachyon-utils-react';
import type { EventTrackerContext } from '../EventTrackerRoot';
import { useEventTrackerContext } from '../EventTrackerRoot';

/**
 * The interface of the options argument passed to createWithTracking. It infers
 * several types based on the values passed to it:
 *
 * - TrackingProps: the props that will be provided by the generated HOC,
 * generally defined and then mixed into the wrapped component's props
 * - ExternalProps: the props that will be exposed on the outside of the
 * generated HOC
 * - InternalProps: the props of the component that will be wrapped by the
 * generated HOC
 * - Payload: the shape of the payload of the event to be tracked
 */
export type CreateWithTrackingOpts<
  TrackingProps,
  ExternalProps,
  InternalProps extends ExternalProps & TrackingProps,
  Payload,
> = {
  /**
   * The component that will be given event tracking functionality
   */
  Comp: ComponentType<InternalProps>;
  /**
   * Curried function for accepting the React context object containing the
   * onEvent function (and other values). It then returns a new function that
   * accepts the target event type's payload and maps that into properly-shaped
   * EventData for that event type.
   */
  createTrackingFunction: (
    context: EventTrackerContext,
  ) => (payload: Payload) => void;
  /**
   * The display name for the generated HOC. For instance, if you provide
   * 'EventTracking', the HOC will have a displayName of
   * "EventTracking(WrappedComponentName)". This can be any value but is
   * recommeded to match whatever the name of the your new HOC function will be:
   * for instance `EventTracking` and `withEventTracking`. This will be used
   * in tests, latency reporting, and other things.
   */
  displayName: string;
  /**
   * The name of the prop function that will be passed into the wrapped
   * component. For instance, if you provide `trackEvent`, the wrapped
   * component will have access to `this.props.trackEvent` for reporting events.
   * This string must be the same as the one defined in your TrackingProps
   * (Typescript will enforce this).
   */
  trackingFunctionName: keyof TrackingProps;
};

/**
 * Function for creating tracking HOCs for specific event types which handles
 * unwrapping context and mapping props into the wrapped component. This should
 * generally be called from within another function that represents the public
 * interface to tracking a specific event type. See withInteractionTracking for
 * usage examples and the docs on the CreateWithTrackingOpts interface for more
 * details on how the arguments work.
 *
 * @param CreateWithTrackingOpts object for provided named arguments
 */
export function createWithTracking<
  TrackingProps,
  ExternalProps,
  InternalProps extends ExternalProps & TrackingProps,
  Payload,
>({
  Comp,
  createTrackingFunction,
  displayName,
  trackingFunctionName,
}: CreateWithTrackingOpts<
  TrackingProps,
  ExternalProps,
  InternalProps,
  Payload
>): FC<ExternalProps> {
  const Trackable: FC<ExternalProps> = (props) => {
    const eventTracker = useEventTrackerContext();

    const newProps = {
      ...props,
      [trackingFunctionName]: createTrackingFunction(eventTracker),
    } as InternalProps;

    return <Comp {...newProps} />;
  };

  Trackable.displayName = `${displayName}(${getComponentName(Comp)})`;

  return Trackable;
}
