import type { ComponentType, FC } from 'react';
import { createWithTracking } from '../createWithTracking';
import type { DefaultCustomPayload, MinimumCustomPayload } from '../event-data';
import type { CustomEventDataPayload, TrackEvent } from './createTrackEvent';
import { createTrackEvent } from './createTrackEvent';

/**
 * Props for mixing into components wrapped by withCustomTracking
 */
export type CustomTrackingProps<
  T extends MinimumCustomPayload = DefaultCustomPayload,
> = {
  /**
   * Function for reporting custom events
   */
  trackEvent: TrackEvent<T>;
};

/**
 * The props of the wrapped component exposed beyond withCustomTracking
 */
type CustomTrackingExternalProps<P> = Omit<P, keyof CustomTrackingProps>;

/**
 * withCustomTracking is a HOC that provides the trackEvent prop function to
 * the wrapped component, allowing it to report arbitrary events. Frequently
 * used event types should ideally be promoted to their own HOC based on
 * createWithTracking to ensure proper event shapes. To use, extend the
 * wrapped components props with CustomTrackingProps. It accepts any keys but
 * reserves `type` for internal use. The event values must be primitive:
 * string | number | boolean | null | undefined.
 *
 * The field `event` will be used for the event type.
 *
 * @param Comp A component in which to track custom events
 */
export function withCustomTracking<P extends CustomTrackingProps>(
  Comp: ComponentType<P>,
): FC<CustomTrackingExternalProps<P>> {
  return createWithTracking<
    CustomTrackingProps,
    CustomTrackingExternalProps<P>,
    P,
    CustomEventDataPayload
  >({
    Comp,
    createTrackingFunction: createTrackEvent,
    displayName: 'CustomTracking',
    trackingFunctionName: 'trackEvent',
  });
}
