import type { DeviceOrientation } from './types';

/**
 * Processed browser details as defined by Data Science
 */
export type BrowserAndOsInfo = {
  browser_family: string;
  browser_version: string;
  os_name: string;
  os_version: string;
};

/**
 * Properties that are calculated at app start and then considered immutable.
 */
export type StaticProperties = BrowserAndOsInfo & {
  /**
   * The ID that uniquely identifies this device's session. This is likely a
   * random value that has a low probability of colliding when used in context
   * to the device ID, client app, & platform.
   */
  app_session_id: string;
  /**
   * The unique version string for the application. This might be a commit hash
   * or build ID.
   */
  app_version: string;
  /**
   * The user agent string available from navigator.userAgent.
   */
  browser: string;
  /**
   * The name of the reporting application. This should be unique to the
   * application so that event receivers can separate events by the reporting
   * app.
   */
  client_app: string;
  /**
   * The unique identifier for this (device, browser). This is normally in the
   * unique_id cookie.
   */
  device_id: string;
  domain: string;
  host: string;
  /**
   * The language the application is presented in.
   */
  language: string;
  /**
   * The platform the application exists on, e.g. web, mobile web, iOS, etc.
   */
  platform: string;
  /**
   * The URL that referred the client to this URL.
   */
  referrer: string;
  referrer_domain: string;
  referrer_host: string;
};

/**
 * Identifying Users
 * https://data.xarth.tv/faq/identifiers.html#the-user-identifiers
 *
 * Spade default Properties
 * https://data.xarth.tv/events/default_properties.html
 */
export type UserTrackingInfo =
  | {
      /**
       * Indicates if the client is logged in or not.
       */
      logged_in: true;
      /*
       * The name of the account of a logged-in user.
       */
      login: string | null;
      /**
       * The id of the account of a logged-in user.
       */
      user_id: string;
    }
  | {
      logged_in: false;
      login: null;
      user_id: null;
    };

/**
 * Properties that can change with each reported event.
 */
export type DynamicProperties = UserTrackingInfo & {
  /**
   * The current time on the client device. Seconds since epoch as float.
   */
  client_time: number;
  /**
   * The in-app location (experiences as defined by )
   */
  location: string | undefined;
  /**
   * The orientation of the device, either portrait or landscape. See
   * DeviceOrientation for more details.
   */
  orientation: DeviceOrientation;
  /**
   * A 32 character ID that is generated on every pageview event.
   */
  page_session_id: string | undefined;
  time: number;
  /**
   * The URL on which the event occurred.
   */
  url: string | undefined;
  /**
   * The height of the viewport in pixels.
   */
  viewport_height: number;
  /**
   * The width of the viewport in pixels.
   */
  viewport_width: number;
};

export type CommonEventProperties = DynamicProperties & StaticProperties;

export type CommonEventData = {
  properties: CommonEventProperties;
};
