import type { CommonEventData } from './common';
import type { EventType } from './types';

/**
 * List of ways a user can interact with a component that we track.
 */
export enum InteractionType {
  /**
   * Click applies to both clicks and taps.
   */
  Click = 'click',
  Impression = 'impression',
}

/**
 * Interaction data as seen in blueprint: https://blueprint.di.xarth.tv/#/schema/ui_interaction
 */
export type InteractionEventProperties = {
  /**
   * The type of interaction triggering event
   */
  interaction: InteractionType;
  /**
   * The content that the user interacted with. This value is saved, unmapped,
   * to Spade. As such, changing this value will create a discontinuity in
   * historical data.
   */
  interaction_content: string;
  /**
   * The medium for the event. This value is built up to indicate the context of
   * the interaction. For example, it might be somePage.somePart.someComponent.
   * This value is saved, unmapped, to the database. Changing it will create
   * a discontinuity with historical data.
   */
  interaction_medium: string;
  /**
   * The path for the page the user will go to as a result of interacting with
   * the component. Ideally, this should be normalizable across platforms.
   */
  interaction_target_path?: string | undefined;
};

/**
 * Data about a UI interaction event.
 */
export type RawInteractionEventData = {
  event: EventType.Interaction;
  properties: InteractionEventProperties;
};

export type ProcessedInteractionEventData = CommonEventData &
  RawInteractionEventData;
