import { lorem } from 'faker';
import type { FC } from 'react';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import type { EventTrackerContext } from '../EventTrackerRoot';
import { eventTrackerContext } from '../EventTrackerRoot';
import { EventType, InteractionType } from '../event-data';
import { ExtendInteractionMedium } from './ExtendInteractionMedium';
import type { InteractionDataPayload } from './createTrackInteraction';
import { useInteractionTracking } from './useInteractionTracking';
import { useTrackedListItem } from './useTrackedListItem';
import type { InteractionTrackingProps } from './withInteractionTracking';
import { withInteractionTracking } from './withInteractionTracking';

describe('Interaction Tracking', () => {
  const MockInteractionTrackingRenderer: FC<EventTrackerContext> = ({
    children,
    ...ctxProps
  }) => {
    return (
      <eventTrackerContext.Provider children={children} value={ctxProps} />
    );
  };

  const setup = createMountWrapperFactory(
    MockInteractionTrackingRenderer,
    () => ({
      interactionMedium: '',
      location: undefined,
      onEvent: jest.fn(),
      twitchTracking: {},
    }),
  );

  const TestElement = withInteractionTracking(
    (props: InteractionTrackingProps) => (
      <div
        id="test-element"
        onClick={() =>
          props.trackInteraction({
            interaction: props.interaction!,
            interactionContent: props.interactionContent!,
            interactionTargetPath: props.interactionTargetPath,
          })
        }
      />
    ),
  );

  const baseProps: InteractionDataPayload = {
    interaction: InteractionType.Click,
    interactionContent: lorem.word(),
    interactionTargetPath: lorem.word(),
  };

  const baseMedium = lorem.word();

  it('gets full interaction payload with medium from context', () => {
    const { props, wrapper } = setup({
      children: <TestElement {...baseProps} />,
      interactionMedium: baseMedium,
    });
    wrapper.find('#test-element').simulate('click');

    expect(props.onEvent).toHaveBeenCalledWith({
      event: EventType.Interaction,
      properties: expect.objectContaining({
        interaction: baseProps.interaction,
        interaction_content: baseProps.interactionContent,
        interaction_medium: baseMedium,
        interaction_target_path: baseProps.interactionTargetPath,
      }),
    });
  });

  it('extended medium gets set when given a string', () => {
    const extension = lorem.word();
    const { props, wrapper } = setup({
      children: (
        <ExtendInteractionMedium value={extension}>
          <TestElement {...baseProps} />
        </ExtendInteractionMedium>
      ),
      interactionMedium: baseMedium,
    });
    wrapper.find('#test-element').simulate('click');

    expect(props.onEvent).toHaveBeenCalledWith({
      event: EventType.Interaction,
      properties: expect.objectContaining({
        interaction: baseProps.interaction,
        interaction_content: baseProps.interactionContent,
        interaction_medium: `${baseMedium}.${extension}`,
        interaction_target_path: baseProps.interactionTargetPath,
      }),
    });
  });

  it('extended medium gets set when given a function', () => {
    const extension = lorem.word();
    const { props, wrapper } = setup({
      children: (
        <ExtendInteractionMedium value={() => extension}>
          <TestElement {...baseProps} />
        </ExtendInteractionMedium>
      ),
      interactionMedium: baseMedium,
    });
    wrapper.find('#test-element').simulate('click');

    expect(props.onEvent).toHaveBeenCalledWith({
      event: EventType.Interaction,
      properties: expect.objectContaining({
        interaction: baseProps.interaction,
        interaction_content: baseProps.interactionContent,
        interaction_medium: `${baseMedium}.${extension}`,
        interaction_target_path: baseProps.interactionTargetPath,
      }),
    });
  });

  it('useInteractionTracking works as expected', () => {
    const TestListElement = (props: { position: number }) => {
      const trackInteraction = useInteractionTracking();
      const { interactionContent } = useTrackedListItem(props.position);

      return (
        <div
          id="test-list-element"
          onClick={() =>
            trackInteraction({
              interaction: InteractionType.Click,
              interactionContent,
              interactionTargetPath: '/foo',
            })
          }
        />
      );
    };

    const { props, wrapper } = setup({
      children: <TestListElement position={1} />,
      interactionMedium: baseMedium,
    });
    wrapper.find('#test-list-element').simulate('click');

    expect(props.onEvent).toHaveBeenCalledWith({
      event: EventType.Interaction,
      properties: expect.objectContaining({
        interaction: InteractionType.Click,
        interaction_content: `position_1`,
        interaction_medium: baseMedium,
        interaction_target_path: '/foo',
      }),
    });
  });
});
