import type { ComponentType, FC } from 'react';
import { createWithTracking } from '../createWithTracking';
import type {
  InteractionDataPayload,
  TrackInteraction,
} from './createTrackInteraction';
import { createTrackInteraction } from './createTrackInteraction';

/**
 * Props for mixing into components wrapped by withInteractionTracking.
 * Includes payload as partial to allow parents to optionally pass in values.
 */
export type InteractionTrackingProps = Partial<InteractionDataPayload> & {
  /**
   * Function for reporting interaction events
   */
  trackInteraction: TrackInteraction;
};

/**
 * The props of the wrapped component exposed beyond withInteractionTracking
 */
type InteractionTrackingExternalProps<P> = Omit<
  P,
  keyof Pick<InteractionTrackingProps, 'trackInteraction'>
>;

/**
 * withInteractionTracking is a HOC that provides the trackInteraction prop
 * function to the wrapped component, allowing it to report interaction events.
 * To use, extend the wrapped components props with InteractionTrackingProps.
 * interactionContent and interactionType are the only mandatory keys for the
 * event shape, but interactionTargetPath can be used to indicate the
 * destination of things like links.
 *
 * @param Comp A component in which to track custom events
 */
export function withInteractionTracking<P extends InteractionTrackingProps>(
  Comp: ComponentType<P>,
): FC<InteractionTrackingExternalProps<P>> {
  return createWithTracking<
    InteractionTrackingProps,
    InteractionTrackingExternalProps<P>,
    P,
    InteractionDataPayload
  >({
    Comp,
    createTrackingFunction: createTrackInteraction,
    displayName: 'InteractionTracking',
    trackingFunctionName: 'trackInteraction',
  });
}
