import { datatype, lorem } from 'faker';
import type {
  DynamicProperties,
  RawCustomEventData,
  RawDeviceCodeFlowEventData,
  RawInteractionEventData,
  RawPageviewEventData,
  StaticProperties,
} from '../event-data';
import { EventType, InteractionType } from '../event-data';
import { getDynamicProperties } from '../utils';
import { processEvent } from '.';

jest.mock('../utils', () => ({
  getDynamicProperties: jest.fn(),
}));
const mockGetDynamicProperties = getDynamicProperties as jest.Mock;

describe(processEvent, () => {
  const getUserTracking = jest.fn();
  // subset for testing
  const dynamicProperties = { client_time: 23 } as DynamicProperties;
  const staticProperties = {
    app_session_id: 'app_session_id',
    app_version: 'app_version',
  } as StaticProperties;

  beforeEach(() => {
    mockGetDynamicProperties.mockReset();
    mockGetDynamicProperties.mockImplementation(() => dynamicProperties);
  });

  it('handles a custom event', () => {
    const location = lorem.word();
    const page_session_id = lorem.word();
    const properties = {
      [lorem.word()]: datatype.boolean(),
      [lorem.word()]: datatype.number(),
      [lorem.word()]: lorem.words(),
    };

    const data: RawCustomEventData = {
      event: EventType.Custom,
      properties: {
        ...properties,
        event: lorem.word(),
      },
    };

    expect(
      processEvent({
        data,
        getUserTracking,
        location,
        page_session_id,
        staticProperties,
      }),
    ).toEqual({
      event: data.properties.event,
      properties: {
        ...dynamicProperties,
        ...staticProperties,
        ...properties,
      },
    });
    expect(mockGetDynamicProperties).toHaveBeenCalledWith({
      getUserTracking,
      location,
      page_session_id,
    });
  });

  it('handles a custom event and preserves custom location and client_time', () => {
    mockGetDynamicProperties.mockImplementationOnce(() => ({
      ...dynamicProperties,
      location: lorem.word(),
    }));

    const location = lorem.word();
    const page_session_id = lorem.word();
    const properties = {
      [lorem.word()]: datatype.boolean(),
      [lorem.word()]: datatype.number(),
      [lorem.word()]: lorem.words(),
      client_time: datatype.number(),
      location: lorem.word(),
    };

    const data: RawCustomEventData = {
      event: EventType.Custom,
      properties: {
        ...properties,
        event: lorem.word(),
      },
    };

    expect(
      processEvent({
        data,
        getUserTracking,
        location,
        page_session_id,
        staticProperties,
      }),
    ).toEqual({
      event: data.properties.event,
      properties: {
        ...dynamicProperties,
        ...staticProperties,
        ...properties,
      },
    });
    expect(mockGetDynamicProperties).toHaveBeenCalledWith({
      getUserTracking,
      location,
      page_session_id,
    });
  });

  it('handles an interaction event', () => {
    const location = lorem.word();
    const page_session_id = lorem.word();
    const properties = {
      interaction: InteractionType.Click,
      interaction_content: lorem.word(),
      interaction_medium: lorem.word(),
    };

    const data: RawInteractionEventData = {
      event: EventType.Interaction,
      properties,
    };

    expect(
      processEvent({
        data,
        getUserTracking,
        location,
        page_session_id,
        staticProperties,
      }),
    ).toEqual({
      event: EventType.Interaction,
      properties: {
        ...properties,
        ...dynamicProperties,
        ...staticProperties,
      },
    });
    expect(mockGetDynamicProperties).toHaveBeenCalledWith({
      getUserTracking,
      location,
      page_session_id,
    });
  });

  it('handles DeviceCodeFlow event', () => {
    const location = lorem.word();
    const page_session_id = lorem.word();
    const properties = {
      context: 'qr_code',
      flow_state: 'activated',
    } as const;

    const data: RawDeviceCodeFlowEventData = {
      event: EventType.DeviceCodeFlow,
      properties,
    };

    expect(
      processEvent({
        data,
        getUserTracking,
        location,
        page_session_id,
        staticProperties,
      }),
    ).toEqual({
      event: EventType.DeviceCodeFlow,
      properties: {
        ...properties,
        ...dynamicProperties,
        ...staticProperties,
      },
    });
    expect(mockGetDynamicProperties).toHaveBeenCalledWith({
      getUserTracking,
      location,
      page_session_id,
    });
  });

  it('handles a pageview event with root location overriding event location', () => {
    const location = lorem.word();
    const page_session_id = lorem.word();
    const properties = {
      channel: lorem.word(),
      channel_id: datatype.number(),
      client_offline: datatype.boolean(),
      content: lorem.word(),
      email_id: datatype.uuid(),
      game: lorem.word(),
      is_live: datatype.boolean(),
      location: lorem.word(),
      medium: lorem.word(),
      vod_id: datatype.uuid(),
      vod_type: undefined,
    };

    const customDynamicProperties = { ...dynamicProperties, location };
    mockGetDynamicProperties.mockImplementation(() => customDynamicProperties);

    const data: RawPageviewEventData = {
      event: EventType.Pageview,
      properties,
    };

    expect(
      processEvent({
        data,
        getUserTracking,
        location,
        page_session_id,
        staticProperties,
      }),
    ).toEqual({
      event: EventType.Pageview,
      properties: {
        ...properties,
        ...customDynamicProperties,
        ...staticProperties,
        location,
      },
    });
    expect(mockGetDynamicProperties).toHaveBeenCalledWith({
      getUserTracking,
      location,
      page_session_id,
    });
  });

  it('handles a pageview event with event location preserved for undefined root location', () => {
    const location = undefined;
    const page_session_id = lorem.word();
    const properties = {
      channel: lorem.word(),
      channel_id: datatype.number(),
      client_offline: datatype.boolean(),
      content: lorem.word(),
      email_id: datatype.uuid(),
      game: lorem.word(),
      is_live: datatype.boolean(),
      location: lorem.word(),
      medium: lorem.word(),
      vod_id: datatype.uuid(),
      vod_type: undefined,
    };

    const customDynamicProperties = { ...dynamicProperties, location };
    mockGetDynamicProperties.mockImplementation(() => customDynamicProperties);

    const data: RawPageviewEventData = {
      event: EventType.Pageview,
      properties,
    };

    expect(
      processEvent({
        data,
        getUserTracking,
        location,
        page_session_id,
        staticProperties,
      }),
    ).toEqual({
      event: EventType.Pageview,
      properties: {
        ...properties,
        ...customDynamicProperties,
        ...staticProperties,
        location: properties.location,
      },
    });
    expect(mockGetDynamicProperties).toHaveBeenCalledWith({
      getUserTracking,
      location,
      page_session_id,
    });
  });
});
