import { isBrowser } from 'tachyon-utils-stdlib';
import { DeviceOrientation } from '../../event-data';

/**
 * Gets the current orientation of a device and returns it as a
 * DeviceOrientation.
 *
 * This function uses the following strategy:
 * 1. If the Orientation API is available use that.
 * 2. If clientWidth & clientHeight are available compare them.
 * 3. Otherwise report landscape.
 *
 * The comparison of clientWidth & clientHeight is:
 * 1. If the width is greater than the height return landscape.
 * 2. Otherwise return portrait.
 */
export function getDeviceOrientation(): DeviceOrientation {
  if (!isBrowser()) {
    return DeviceOrientation.Server;
  }

  if (window.screen?.orientation?.type) {
    return window.screen.orientation.type.split('-')[0] as DeviceOrientation;
  } else if (typeof window.orientation === 'number') {
    const orientation = Math.abs(window.orientation);
    if (45 < orientation && orientation < 135) {
      return DeviceOrientation.Landscape;
    }
    return DeviceOrientation.Portrait;
  }
  const width = document.documentElement?.clientWidth;
  const height = document.documentElement?.clientHeight;

  if (!width || !height) {
    return DeviceOrientation.Landscape;
  }

  return width > height
    ? DeviceOrientation.Landscape
    : DeviceOrientation.Portrait;
}
