import { isBrowser } from 'tachyon-utils-stdlib';
import type { DynamicProperties, UserTrackingInfo } from '../../event-data';
import { getDeviceOrientation } from '../getDeviceOrientation';

export type UserTracking = {
  id: string;
  login: string | null;
};

export type GetDynamicPropertiesOpts = Pick<
  DynamicProperties,
  'location' | 'page_session_id'
> & {
  /**
   * Used to populate tracking fields pertaining to the current user.
   */
  getUserTracking?: (() => UserTracking | null | undefined) | undefined;
};

/**
 * Properties that represent dynamic properties to be sent to Spade.
 * **NOTE**: This field can change with *each* call, meaning that
 * you cannot assume two calls will be shallow or deeply equal.
 */
export function getDynamicProperties({
  getUserTracking,
  location,
  page_session_id,
}: GetDynamicPropertiesOpts): DynamicProperties {
  const time = Date.now() / 1000;
  const user = getUserTracking?.();
  const userTracking: UserTrackingInfo = user
    ? {
        logged_in: true,
        login: user.login,
        user_id: user.id,
      }
    : {
        logged_in: false,
        login: null,
        user_id: null,
      };

  const browserProperties = isBrowser()
    ? {
        url: window.document.location?.href,
        viewport_height: window.innerHeight,
        viewport_width: window.innerWidth,
      }
    : {
        url: '',
        viewport_height: 0,
        viewport_width: 0,
      };

  return {
    ...userTracking,
    ...browserProperties,
    client_time: time,
    location,
    orientation: getDeviceOrientation(),
    page_session_id,
    time,
  };
}
