import { datatype, lorem } from 'faker';
import { isBrowser } from 'tachyon-utils-stdlib';
import type { UserTracking } from '.';
import { getDynamicProperties } from '.';

jest.mock('tachyon-utils-stdlib', () => ({
  isBrowser: jest.fn(),
}));
const mockIsBrowser = isBrowser as jest.Mock;

describe(getDynamicProperties, () => {
  const location = lorem.word();
  const page_session_id = lorem.word();

  it('passes through location and page_session_id', () => {
    expect(getDynamicProperties({ location, page_session_id })).toEqual(
      expect.objectContaining({ location, page_session_id }),
    );
  });

  describe('user handling', () => {
    it('extracts info from a user when it gets one', () => {
      const user: UserTracking = {
        id: 'id',
        login: 'login',
      };
      const getUserTracking = () => user;

      expect(
        getDynamicProperties({ getUserTracking, location, page_session_id }),
      ).toEqual(
        expect.objectContaining({
          logged_in: true,
          login: 'login',
          user_id: 'id',
        }),
      );
    });

    it('falls back to default when it does not get a user', () => {
      const getUserTracking = () => undefined;

      expect(
        getDynamicProperties({ getUserTracking, location, page_session_id }),
      ).toEqual(
        expect.objectContaining({
          logged_in: false,
          login: null,
          user_id: null,
        }),
      );
    });

    it('falls back to default when user tracking not provided', () => {
      expect(getDynamicProperties({ location, page_session_id })).toEqual(
        expect.objectContaining({
          logged_in: false,
          login: null,
          user_id: null,
        }),
      );
    });
  });

  describe('browser properties', () => {
    it('reports values in the browser', () => {
      mockIsBrowser.mockImplementationOnce(() => true);
      const viewport_height = datatype.number({ min: 1 });
      const viewport_width = datatype.number({ min: 1 });

      const { innerHeight, innerWidth } = window;
      (window.innerWidth as any) = viewport_width;
      (window.innerHeight as any) = viewport_height;

      expect(getDynamicProperties({ location, page_session_id })).toEqual(
        expect.objectContaining({
          url: 'https://m.twitch.tv/',
          viewport_height,
          viewport_width,
        }),
      );

      (window.innerWidth as any) = innerWidth;
      (window.innerHeight as any) = innerHeight;
    });

    it('reports fallbacks on the server', () => {
      mockIsBrowser.mockImplementationOnce(() => false);

      expect(getDynamicProperties({ location, page_session_id })).toEqual(
        expect.objectContaining({
          url: '',
          viewport_height: 0,
          viewport_width: 0,
        }),
      );
    });
  });
});
