import type { DataScienceNetworkInformation } from 'tachyon-type-library';
import { isBrowser } from 'tachyon-utils-stdlib';
import type { StaticProperties } from '../../event-data';
import { getBrowserAndOsInfo } from '../getBrowserAndOsInfo';
import { getReferrerDomain, getReferrerHost } from '../referrer';

export type GetStaticPropertiesOpts = {
  /**
   * The unique version string for the application. This might be a commit hash
   * or build ID.
   */
  appVersion: string;
  /**
   * The name of the reporting application. This should be unique to the
   * application so that event receivers can separate events by the reporting
   * app.
   */
  clientApp: string;
  /**
   * The unique identifier for this (device, browser). This is normally in the
   * unique_id cookie.
   */
  deviceID: string;
  /**
   * The language the application is presented in.
   */
  language: string;
  netInfo: DataScienceNetworkInformation;
  /**
   * The platform the application exists on, e.g. web, mobile web, iOS, etc.
   */
  platform: string;
  /**
   * The ID that uniquely identifies this device's session. This is likely a
   * random value that has a low probability of colliding when used in context
   * to the device ID, client app, & platform.
   */
  sessionID: string;
};

export function getStaticProperties({
  appVersion,
  clientApp,
  deviceID,
  language,
  netInfo,
  platform,
  sessionID,
}: GetStaticPropertiesOpts): StaticProperties {
  const browserProperties = isBrowser()
    ? {
        browser: window.navigator.userAgent,
        domain: window.location.hostname,
        host: `${window.location.protocol}//${window.location.hostname}`,
        referrer: window.document.referrer || '',
        referrer_domain: getReferrerDomain(),
        referrer_host: getReferrerHost(),
      }
    : {
        browser: '',
        domain: '',
        host: '',
        referrer: '',
        referrer_domain: '',
        referrer_host: '',
      };

  return {
    ...netInfo,
    ...browserProperties,
    ...getBrowserAndOsInfo(browserProperties.browser),
    app_session_id: sessionID,
    app_version: appVersion,
    client_app: clientApp,
    device_id: deviceID,
    language: language.toLowerCase(),
    platform,
    // these are in the spec but were unimplemented in legacy mobile web
    // preferred_language: preferredLanguage,
    // received_language: receivedLanguage,
  };
}
