import { datatype, lorem } from 'faker';
import type { DataScienceNetworkInformation } from 'tachyon-type-library';
import { ConnectionType, EffectiveConnectionType } from 'tachyon-type-library';
import { isBrowser } from 'tachyon-utils-stdlib';
import { getReferrerDomain, getReferrerHost } from '../referrer';
import type { GetStaticPropertiesOpts } from '.';
import { getStaticProperties } from '.';

jest.mock('tachyon-utils-stdlib', () => ({
  ...jest.requireActual('tachyon-utils-stdlib'),
  isBrowser: jest.fn(),
}));
const mockIsBrowser = isBrowser as jest.Mock;

jest.mock('../referrer', () => ({
  getReferrerDomain: jest.fn(),
  getReferrerHost: jest.fn(),
}));
const mockGetReferrerDomain = getReferrerDomain as jest.Mock;
const mockGetReferrerHost = getReferrerHost as jest.Mock;

describe(getStaticProperties, () => {
  beforeEach(() => {
    mockGetReferrerDomain.mockReset();
    mockGetReferrerHost.mockReset();
  });

  const netInfo: DataScienceNetworkInformation = {
    bandwidth: datatype.number(),
    bandwidth_max: datatype.number(),
    effective_mobile_connection_type: EffectiveConnectionType.FourthGeneration,
    mobile_connection_type: ConnectionType.Cellular,
    round_trip_time: datatype.number(),
  };

  const opts: GetStaticPropertiesOpts = {
    appVersion: 'appVersion',
    clientApp: 'clientApp',
    deviceID: 'deviceID',
    language: 'Language',
    netInfo,
    platform: 'platform',
    sessionID: 'sessionID',
  };

  it('lowercases language value', () => {
    expect(getStaticProperties(opts)).toEqual(
      expect.objectContaining({
        language: 'language',
      }),
    );
  });

  it('returns browser properties in browser', () => {
    const referrer_domain = lorem.word();
    const referrer_host = lorem.word();
    mockIsBrowser.mockImplementationOnce(() => true);
    mockGetReferrerDomain.mockImplementationOnce(() => referrer_domain);
    mockGetReferrerHost.mockImplementationOnce(() => referrer_host);

    expect(getStaticProperties(opts)).toEqual(
      expect.objectContaining({
        browser: expect.stringContaining('Mozilla'),
        domain: 'm.twitch.tv',
        host: 'https://m.twitch.tv',
        referrer: '',
        referrer_domain,
        referrer_host,
      }),
    );
  });

  it('returns browser property fallbacks on server', () => {
    mockIsBrowser.mockImplementationOnce(() => false);
    expect(getStaticProperties(opts)).toEqual(
      expect.objectContaining({
        browser: '',
        domain: '',
        host: '',
        referrer: '',
        referrer_domain: '',
        referrer_host: '',
      }),
    );
  });
});
