import type { FC, ReactNode } from 'react';
import { createContext, useContext, useEffect } from 'react';
import { ExperimentGroup } from '../experimentInfo';
import { useExperimentsRootContext } from './ExperimentsRoot';

/**
 * Private context used within this package for communicating between
 * Experiment and Treatment components.
 */
export const experimentsActiveGroupContext = createContext<ExperimentGroup>(
  ExperimentGroup.Control,
);

export const useExperimentGroupIsActive = (group: ExperimentGroup): boolean =>
  group === useContext(experimentsActiveGroupContext);

export interface ExperimentProps {
  children?: ReactNode;
  /**
   * The uuid of the experiment being conducted on the user.
   */
  experimentUUID: string;
  /**
   * Set this to true if for some reason you don't want the branch decision
   * reported to Spade. This should not be used simply for deduping as the
   * framework takes care of that itself.
   */
  preventReportingBranch?: boolean;
}

/**
 * The Experiment component is how you declare an experiment. It takes the
 * name of the experiment through `experimentName` and then will render the
 * appropriate treatment based on the experiment assignment.
 *
 * You can optionally disable reporting with `preventReportingBranch`.
 */
export const Experiment: FC<ExperimentProps> = ({
  children,
  experimentUUID,
  preventReportingBranch,
}) => {
  const { experimentGroupsForUser, onExperimentBranch } =
    useExperimentsRootContext();
  const experimentData = experimentGroupsForUser[experimentUUID];

  useEffect(() => {
    // Don't report if we're missing experiment data
    // https://docs.google.com/document/d/1LIKBlbDaA9KQLrEUpSpMQ2NECww1JkKaz8M2CZPB4QQ/edit#heading=h.uae6wug3njjy
    if (experimentData && !preventReportingBranch) {
      onExperimentBranch(experimentData);
    }
  }, [experimentData, onExperimentBranch, preventReportingBranch]);

  return (
    <experimentsActiveGroupContext.Provider
      children={children}
      value={experimentData?.group ?? ExperimentGroup.Control}
    />
  );
};

Experiment.displayName = 'Experiment';
