import type { FC } from 'react';
import { createContext, useCallback, useContext, useMemo, useRef } from 'react';
import type { ExperimentGroup } from '../experimentInfo';
import type {
  ExperimentGroupsForUser,
  ExperimentSettings,
  UserExperimentMetadata,
} from '../utils';
import { getExperimentGroupsForUser } from '../utils';

/**
 * The name of the event sent to Spade when a treatment is decided by
 * the client for an experiment.
 */
export const EXPERIMENT_BRANCH_EVENT_NAME = 'experiment_branch';

/**
 * Properties unique to the experiment branch event as defined in blueprint:
 * https://blueprint.di.xarth.tv/#/schema/experiment_branch
 *
 * Further elaboration here:
 * https://docs.google.com/document/d/17UOcXoXfd3hjEwvdj461rY42hSLqsuf9NrWfahS9PRU/
 */
export type ExperimentBranchEvent = {
  event: typeof EXPERIMENT_BRANCH_EVENT_NAME;
  /**
   * Group returned by the call to minixperiment.js for this user in this
   * experiment.
   */
  experiment_group: ExperimentGroup;
  /**
   * UUID of experiment, from experiments.json.
   */
  experiment_id: string;
  /**
   * Name of experiment, from experiments.json.
   */
  experiment_name: string;
  /**
   * Unit of enrollment for this experiment, from experiments.json.
   */
  experiment_type: 'device_id';
  /**
   * Version of experiment being executed, from experiments.json.
   */
  experiment_version: number;
  /**
   * experiment seed of current user
   */
  request_id: string;
};

export type ExperimentsRootContext = {
  /**
   * This field contains a map of all experiment names to the data about that
   * experiment.
   */
  readonly experimentGroupsForUser: ExperimentGroupsForUser;
  /**
   * A callback to handle an experiment branch event.
   */
  readonly onExperimentBranch: (experimentData: UserExperimentMetadata) => void;
};

/**
 * Public context used within this package for communicating between Root and
 * Experiment components and exposing ExperimentGroupsForUser to apps dangerously.
 */
const experimentsRootContext = createContext<ExperimentsRootContext>({
  experimentGroupsForUser: {},
  onExperimentBranch: () => null,
});

export const useExperimentsRootContext = (): ExperimentsRootContext =>
  useContext(experimentsRootContext);

export type ExperimentsRootProps = ExperimentSettings & {
  /**
   * Function provided for consuming experiment branch events that are
   * generated. For instance, this could be an event reporting function.
   */
  onEvent: (event: ExperimentBranchEvent) => void;
};

/**
 * ExperimentsRoot initializes an experiment system for a React application,
 * and must be put in the React tree above all other experiment functionality
 * from this package.
 */
export const ExperimentsRoot: FC<ExperimentsRootProps> = ({
  bucket,
  children,
  debug,
  experimentOverrides,
  experimentSlots,
  experiments,
  onEvent,
}) => {
  const experimentGroupsForUser = useMemo(
    () =>
      getExperimentGroupsForUser({
        bucket,
        debug,
        experimentOverrides,
        experimentSlots,
        experiments,
      }),
    [bucket, debug, experimentOverrides, experimentSlots, experiments],
  );

  const recordedDecisions = useRef<Set<string>>(new Set());

  /**
   * This method checks to see if this is the first time an experiment branch
   * has been acted upon and, if so, triggers the onEvent event handler.
   */
  const onExperimentBranch = useCallback(
    (experimentData: UserExperimentMetadata): void => {
      if (!recordedDecisions.current.has(experimentData.name)) {
        recordedDecisions.current.add(experimentData.name);
        onEvent({
          event: EXPERIMENT_BRANCH_EVENT_NAME,
          experiment_group: experimentData.group,
          experiment_id: experimentData.uuid,
          experiment_name: experimentData.name,
          experiment_type: experimentData.type,
          experiment_version: experimentData.version,
          request_id: bucket ?? 'missing_bucket',
        });
      }
    },
    [bucket, onEvent, recordedDecisions],
  );

  const contextValue = useMemo(
    () => ({ experimentGroupsForUser, onExperimentBranch }),
    [experimentGroupsForUser, onExperimentBranch],
  );

  return (
    <experimentsRootContext.Provider value={contextValue}>
      {children}
    </experimentsRootContext.Provider>
  );
};

ExperimentsRoot.displayName = 'ExperimentsRoot';
