import { createMountWrapperFactory } from 'tachyon-test-utils';
import { ExperimentGroup } from '../experimentInfo';
import type { UserExperimentMetadata } from '../utils';
import { getExperimentGroupsForUser } from '../utils';
import { Experiment } from './Experiment';
import {
  EXPERIMENT_BRANCH_EVENT_NAME,
  ExperimentsRoot,
} from './ExperimentsRoot';
import { Control, Treatment } from './Group';

jest.mock('../utils', () => ({
  ...jest.requireActual('../utils'),
  getExperimentGroupsForUser: jest.fn(),
}));

const mockGetExperimentGroupsForUser = getExperimentGroupsForUser as jest.Mock;

describe('Experiments', () => {
  const mockExperimentUUID = 'foo';
  const mockExperimentGroupForUser: UserExperimentMetadata = {
    group: ExperimentGroup.Control,
    name: 'mobile_web_foo',
    type: 'device_id' as const,
    uuid: mockExperimentUUID,
    version: 1,
  };
  const bucket = 'a';
  mockGetExperimentGroupsForUser.mockImplementation(() => ({
    [mockExperimentUUID]: mockExperimentGroupForUser,
  }));

  const setup = createMountWrapperFactory(ExperimentsRoot, () => ({
    bucket,
    debug: false,
    experimentOverrides: undefined,
    experimentSlots: { 0: mockExperimentUUID, 1: undefined, 2: undefined },
    experiments: {},
    onEvent: jest.fn(),
  }));

  it('renders control group if the experiment group is Control and reports branching', () => {
    mockGetExperimentGroupsForUser.mockImplementationOnce(() => ({
      [mockExperimentUUID]: {
        ...mockExperimentGroupForUser,
        group: ExperimentGroup.Control,
      },
    }));

    const {
      props: { onEvent },
      wrapper,
    } = setup({
      children: (
        <Experiment experimentUUID={mockExperimentUUID}>
          <Control>
            <div className="control" />
          </Control>
          <Treatment>
            <div className="treatment" />
          </Treatment>
        </Experiment>
      ),
    });

    expect(wrapper.find('.control')).toHaveLength(1);
    expect(wrapper.find('.treatment')).toHaveLength(0);
    expect(onEvent).toHaveBeenCalledTimes(1);
    expect(onEvent).toHaveBeenCalledWith({
      event: EXPERIMENT_BRANCH_EVENT_NAME,
      experiment_group: ExperimentGroup.Control,
      experiment_id: mockExperimentUUID,
      experiment_name: mockExperimentGroupForUser.name,
      experiment_type: 'device_id',
      experiment_version: mockExperimentGroupForUser.version,
      request_id: bucket,
    });
  });

  it('renders treatment group if the experiment group is Treatment and reports branching', () => {
    mockGetExperimentGroupsForUser.mockImplementationOnce(() => ({
      [mockExperimentUUID]: {
        ...mockExperimentGroupForUser,
        group: ExperimentGroup.Treatment,
      },
    }));

    const {
      props: { onEvent },
      wrapper,
    } = setup({
      children: (
        <Experiment experimentUUID={mockExperimentUUID}>
          <Control>
            <div className="control" />
          </Control>
          <Treatment>
            <div className="treatment" />
          </Treatment>
        </Experiment>
      ),
    });

    expect(wrapper.find('.control')).toHaveLength(0);
    expect(wrapper.find('.treatment')).toHaveLength(1);
    expect(onEvent).toHaveBeenCalledTimes(1);
    expect(onEvent).toHaveBeenCalledWith({
      event: EXPERIMENT_BRANCH_EVENT_NAME,
      experiment_group: ExperimentGroup.Treatment,
      experiment_id: mockExperimentUUID,
      experiment_name: mockExperimentGroupForUser.name,
      experiment_type: 'device_id',
      experiment_version: mockExperimentGroupForUser.version,
      request_id: bucket,
    });
  });

  it('does not report branching if preventReportingBranch is true', () => {
    const {
      props: { onEvent },
      wrapper,
    } = setup({
      children: (
        <Experiment experimentUUID={mockExperimentUUID} preventReportingBranch>
          <Control>
            <div className="control" />
          </Control>
          <Treatment>
            <div className="treatment" />
          </Treatment>
        </Experiment>
      ),
    });

    expect(wrapper.find('.control')).toHaveLength(1);
    expect(wrapper.find('.treatment')).toHaveLength(0);
    expect(onEvent).not.toHaveBeenCalled();
  });

  it('renders multiple groups for same experiment but does not report branching twice', () => {
    const {
      props: { onEvent },
      wrapper,
    } = setup({
      children: (
        <>
          <Experiment experimentUUID={mockExperimentUUID}>
            <Control>
              <div className="control" />
            </Control>
            <Treatment>
              <div className="treatment" />
            </Treatment>
          </Experiment>
          <Experiment experimentUUID={mockExperimentUUID}>
            <Control>
              <div className="control" />
            </Control>
            <Treatment>
              <div className="treatment" />
            </Treatment>
          </Experiment>
        </>
      ),
    });

    expect(wrapper.find('.control')).toHaveLength(2);
    expect(wrapper.find('.treatment')).toHaveLength(0);
    expect(onEvent).toHaveBeenCalledTimes(1);
    expect(onEvent).toHaveBeenCalledWith({
      event: EXPERIMENT_BRANCH_EVENT_NAME,
      experiment_group: ExperimentGroup.Control,
      experiment_id: mockExperimentUUID,
      experiment_name: mockExperimentGroupForUser.name,
      experiment_type: 'device_id',
      experiment_version: mockExperimentGroupForUser.version,
      request_id: bucket,
    });
  });

  it('reports branching even without Group children (for non-UI experiments)', () => {
    const {
      props: { onEvent },
    } = setup({
      children: <Experiment experimentUUID={mockExperimentUUID} />,
    });

    expect(onEvent).toHaveBeenCalledTimes(1);
    expect(onEvent).toHaveBeenCalledWith({
      event: EXPERIMENT_BRANCH_EVENT_NAME,
      experiment_group: ExperimentGroup.Control,
      experiment_id: mockExperimentUUID,
      experiment_name: mockExperimentGroupForUser.name,
      experiment_type: 'device_id',
      experiment_version: mockExperimentGroupForUser.version,
      request_id: bucket,
    });
  });

  it('does not report branching when experiment data is missing', () => {
    mockGetExperimentGroupsForUser.mockImplementationOnce(() => ({
      [mockExperimentUUID]: undefined,
    }));

    const {
      props: { onEvent },
    } = setup({
      children: <Experiment experimentUUID={mockExperimentUUID} />,
    });

    expect(onEvent).toHaveBeenCalledTimes(0);
  });
});
