// these should match the values in
// https://minixperiment.di.xarth.tv

// Currently this pacakge is hard-coded to assume all experiments are binary
// A/B tests with groups control/treatment (https://jira.twitch.com/browse/MWC-1098)
/**
 * Valid experiment groups for this experiment framework.
 */
export enum ExperimentGroup {
  /**
   * The group that receives the original experience.
   */
  Control = 'control',
  /**
   * The group that receives the special experience.
   */
  Treatment = 'treatment',
}

/**
 * A static edge dictionary contains the following mappings. This dictionary has 1000 entries mapping to 9 “named” buckets:
 * A: 1% of traffic, all sent to the mweb canary; this bucket will correspond to all control
 * B: 11.5% of traffic, sent to mweb production; this bucket will correspond to all control
 * C, D, E, F, G, H, I: 12.5% of traffic each, sent to mweb production; these buckets will correspond to various control-treatment variations
 *
 * https://docs.google.com/document/d/1XLbVA8M7potkCtwfsRzcnnoCvU7uTyfrCy8Do5SEuSU/edit#heading=h.9c1tikph1s49
 */
export const BUCKET_ASSIGNMENT = {
  a: [
    ExperimentGroup.Control,
    ExperimentGroup.Control,
    ExperimentGroup.Control,
  ],
  b: [
    ExperimentGroup.Control,
    ExperimentGroup.Control,
    ExperimentGroup.Control,
  ],
  c: [
    ExperimentGroup.Control,
    ExperimentGroup.Control,
    ExperimentGroup.Treatment,
  ],
  d: [
    ExperimentGroup.Control,
    ExperimentGroup.Treatment,
    ExperimentGroup.Control,
  ],
  e: [
    ExperimentGroup.Treatment,
    ExperimentGroup.Control,
    ExperimentGroup.Control,
  ],
  f: [
    ExperimentGroup.Control,
    ExperimentGroup.Treatment,
    ExperimentGroup.Treatment,
  ],
  g: [
    ExperimentGroup.Treatment,
    ExperimentGroup.Control,
    ExperimentGroup.Treatment,
  ],
  h: [
    ExperimentGroup.Treatment,
    ExperimentGroup.Treatment,
    ExperimentGroup.Control,
  ],
  i: [
    ExperimentGroup.Treatment,
    ExperimentGroup.Treatment,
    ExperimentGroup.Treatment,
  ],
} as const;

export type ExperimentBucket = keyof typeof BUCKET_ASSIGNMENT;

/**
 * There may only be up to 3 experiments at a time, and their slot
 * must not change for the lieftime of the experiment.
 *
 * Maps slot to UUID.
 */
export type ExperimentSlots = {
  0: string | undefined;
  1: string | undefined;
  2: string | undefined;
};

export type ExperimentSlot = keyof ExperimentSlots;

/**
 * The definition used for weighted experiments.
 */
export type WeightedExperimentGroup = {
  /**
   * What group the subject has been bucketed into.
   */
  readonly value: ExperimentGroup;
  /**
   * The weighting for this experiment. The current implementation only support 0/100 50/50 and 100/0 splits.
   */
  readonly weight: 0 | 50 | 100;
};

/**
 * Metadata that defines everything about an experiment.
 */
export interface ExperimentMetadata {
  /**
   * The possible experiences and their weighting.
   */
  readonly groups: WeightedExperimentGroup[];
  /**
   * The human readable experiment name.
   */
  readonly name: string;
  /**
   * What the experiment is sticky to.
   */
  readonly type: 'device_id';
  /**
   * The UUID for the experiment.
   */
  readonly uuid: string;
  /**
   * The version of the experiment we are using. This is defined by
   * minixperiment and sent to Spade to validate that the treatment logic was
   * up to date.
   */
  readonly version: number;
}
