import type { ExperimentBucket, ExperimentSlot } from '../../experimentInfo';
import { ExperimentGroup } from '../../experimentInfo';
import { generateTestExperimentMetadata } from '../test-utils';
import { getExperimentGroup } from '.';

describe(getExperimentGroup, () => {
  it('control when missing all groups', () => {
    expect(
      getExperimentGroup(
        {
          groups: [],
          name: 'mobile_web_experiment_1',
          type: 'device_id',
          uuid: 'foo',
          version: 1,
        },
        'a',
        0,
      ),
    ).toEqual(ExperimentGroup.Control);
  });

  it('control when missing treatment group', () => {
    expect(
      getExperimentGroup(
        {
          groups: [
            {
              value: ExperimentGroup.Control,
              weight: 50,
            },
          ],
          name: 'mobile_web_experiment_1',
          type: 'device_id',
          uuid: 'foo',
          version: 1,
        },
        'a',
        0,
      ),
    ).toEqual(ExperimentGroup.Control);
  });

  it('control when treatment group is not enabled', () => {
    expect(
      getExperimentGroup(
        generateTestExperimentMetadata({
          control: 50,
          treatment: 0,
          uuid: 'foo',
        }),
        'a',
        0,
      ),
    ).toEqual(ExperimentGroup.Control);
  });

  it('treatment when missing control group', () => {
    expect(
      getExperimentGroup(
        {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 50,
            },
          ],
          name: 'mobile_web_experiment_1',
          type: 'device_id',
          uuid: 'foo',
          version: 1,
        },
        'a',
        0,
      ),
    ).toEqual(ExperimentGroup.Treatment);
  });

  it('treatment when control not enabled', () => {
    expect(
      getExperimentGroup(
        generateTestExperimentMetadata({
          control: 0,
          treatment: 50,
          uuid: 'foo',
        }),
        'a',
        0,
      ),
    ).toEqual(ExperimentGroup.Treatment);
  });

  it.each([
    ['a', 0, ExperimentGroup.Control],
    ['a', 1, ExperimentGroup.Control],
    ['a', 2, ExperimentGroup.Control],
    ['b', 0, ExperimentGroup.Control],
    ['b', 1, ExperimentGroup.Control],
    ['b', 2, ExperimentGroup.Control],
    ['c', 0, ExperimentGroup.Control],
    ['c', 1, ExperimentGroup.Control],
    ['c', 2, ExperimentGroup.Treatment],
    ['d', 0, ExperimentGroup.Control],
    ['d', 1, ExperimentGroup.Treatment],
    ['d', 2, ExperimentGroup.Control],
    ['e', 0, ExperimentGroup.Treatment],
    ['e', 1, ExperimentGroup.Control],
    ['e', 2, ExperimentGroup.Control],
    ['f', 0, ExperimentGroup.Control],
    ['f', 1, ExperimentGroup.Treatment],
    ['f', 2, ExperimentGroup.Treatment],
    ['g', 0, ExperimentGroup.Treatment],
    ['g', 1, ExperimentGroup.Control],
    ['g', 2, ExperimentGroup.Treatment],
    ['h', 0, ExperimentGroup.Treatment],
    ['h', 1, ExperimentGroup.Treatment],
    ['h', 2, ExperimentGroup.Control],
    ['i', 0, ExperimentGroup.Treatment],
    ['i', 1, ExperimentGroup.Treatment],
    ['i', 2, ExperimentGroup.Treatment],
  ])(
    "it determines the experiment group for bucket '%s' slot %s",
    (bucket, slot, expected) => {
      const experiment = generateTestExperimentMetadata({
        control: 50,
        treatment: 50,
        uuid: 'foo',
      });
      const group = getExperimentGroup(
        experiment,
        bucket as ExperimentBucket,
        slot as ExperimentSlot,
      );
      expect(group).toEqual(expected);
    },
  );
});
