import type { ParsedUrlQuery } from 'querystring';
import type { Mutable } from 'type-fest';
import type {
  ExperimentBucket,
  ExperimentGroup,
  ExperimentMetadata,
  ExperimentSlot,
  ExperimentSlots,
} from '../../experimentInfo';
import { getExperimentGroup } from '../getExperimentGroup';
import { parseExperimentOverridesWithWarnings } from '../parseExperimentOverrides';
import type { Minixperiments } from '../processMinixperiments';
import {
  processMinixperiments,
  processMinixperimentsWithWarnings,
} from '../processMinixperiments';

export type UserExperimentMetadata = Omit<ExperimentMetadata, 'groups'> & {
  group: ExperimentGroup;
};

/**
 * A Record of all experiment experiences for a specific user/seed. The keys
 * are the experiment uuid.
 */
export type ExperimentGroupsForUser = Readonly<{
  [uuid: string]: UserExperimentMetadata | undefined;
}>;

/**
 * Named parameters for `getExperimentGroupsForUser`.
 */
export type GetExperimentGroupsForUserOpts = {
  bucket: ExperimentBucket | undefined;
  /**
   * Enables console warnings and inverts experiments that are running at 100% treatment.
   *
   * Should NOT be used in production because of the above mentioned inversion.
   */
  debug?: boolean | undefined;
  /**
   * The raw query param value from `experiment_overrides`. Can either be a
   * single string if values were input with comma separation, or a string
   * array if multiple instances of the query param were passed
   */
  experimentOverrides: ParsedUrlQuery[string] | undefined;
  /*
   * There may only be up to 3 experiments at a time, and their slot
   * must not change for the lieftime of the experiment.
   *
   * Maps slot to UUID.
   */
  experimentSlots: ExperimentSlots;
  /**
   * Dictionary of ExperimentUUID => ExperimentMetadata.
   */
  experiments: Minixperiments | undefined;
};

/**
 * This function fetches the experiment list, calculates the experience based
 * on `userExperimentSeed`, and applies any overrides according to
 * `overrideQueryParams`.
 *
 * @param param0 Contains named params userExperimentSeed & overrideQueryParams.
 */
export function getExperimentGroupsForUser({
  bucket,
  debug = false,
  experimentOverrides,
  experimentSlots,
  experiments,
}: GetExperimentGroupsForUserOpts): ExperimentGroupsForUser {
  if (!bucket) {
    return {};
  }
  const slots = Object.entries(experimentSlots).reduce<{
    [uuid: string]: ExperimentSlot;
  }>((acc, [slot, uuid]) => {
    if (uuid) {
      acc[uuid] = slot as unknown as ExperimentSlot;
    }
    return acc;
  }, {});

  const activeExperiments = Object.keys(slots).reduce<Partial<Minixperiments>>(
    (acc, uuid) => {
      acc[uuid] = experiments?.[uuid];
      return acc;
    },
    {},
  );

  // This is disabled in production as it can cause cache
  // poisoning.
  const overrides =
    debug && experimentOverrides
      ? parseExperimentOverridesWithWarnings(experimentOverrides)
      : {};

  const experimentsForApp = debug
    ? processMinixperimentsWithWarnings(activeExperiments)
    : processMinixperiments(activeExperiments);

  return experimentsForApp.reduce<Mutable<ExperimentGroupsForUser>>(
    (acc, experiment) => {
      acc[experiment.uuid] = {
        group:
          overrides[experiment.uuid] ??
          getExperimentGroup(experiment, bucket, slots[experiment.uuid]),
        name: experiment.name,
        type: experiment.type,
        uuid: experiment.uuid,
        version: experiment.version,
      };
      return acc;
    },
    {},
  );
}
