import { ExperimentGroup } from '../../experimentInfo';
import { getExperimentGroupsForUser } from '.';

describe(getExperimentGroupsForUser, () => {
  const mockUUID = 'foo';

  describe('when no experiment slots are registered', () => {
    it('returns empty object', () => {
      expect(
        getExperimentGroupsForUser({
          bucket: 'a',
          experimentOverrides: '',
          experimentSlots: {
            0: undefined,
            1: undefined,
            2: undefined,
          },
          experiments: {},
        }),
      ).toEqual({});
    });
  });

  describe('when no experiments match registered slot', () => {
    it('returns empty object', () => {
      expect(
        getExperimentGroupsForUser({
          bucket: 'a',
          experimentOverrides: '',
          experimentSlots: {
            0: mockUUID,
            1: undefined,
            2: undefined,
          },
          experiments: {},
        }),
      ).toEqual({});
    });
  });

  describe('when experiment slots and corresponding experiment are provided', () => {
    const mockExperimentMetadata = {
      groups: [
        {
          value: ExperimentGroup.Control,
          weight: 0,
        },
        {
          value: ExperimentGroup.Treatment,
          weight: 100,
        },
      ],
      mockUUID,
      name: 'mobile_web_experiment_1',
      t: 1,
      v: 1,
    };

    it('maps metadata', () => {
      expect(
        getExperimentGroupsForUser({
          bucket: 'a',
          experimentOverrides: '',
          experimentSlots: {
            0: mockUUID,
            1: undefined,
            2: undefined,
          },
          experiments: {
            [mockUUID]: mockExperimentMetadata,
          },
        }),
      ).toEqual({
        [mockUUID]: {
          group: ExperimentGroup.Treatment,
          name: mockExperimentMetadata.name,
          type: 'device_id',
          uuid: mockUUID,
          version: 1,
        },
      });
    });

    it('returns an empty object when bucket is missing', () => {
      expect(
        getExperimentGroupsForUser({
          bucket: undefined,
          experimentOverrides: '',
          experimentSlots: {
            0: mockUUID,
            1: undefined,
            2: undefined,
          },
          experiments: {
            [mockUUID]: mockExperimentMetadata,
          },
        }),
      ).toEqual({});
    });

    describe('overrides', () => {
      it('does not apply overrides when debug is not enabled', () => {
        expect(
          getExperimentGroupsForUser({
            bucket: 'a',
            experimentOverrides: `${mockUUID}=${ExperimentGroup.Control}`,
            experimentSlots: {
              0: mockUUID,
              1: undefined,
              2: undefined,
            },
            experiments: {
              [mockUUID]: mockExperimentMetadata,
            },
          }),
        ).toEqual({
          [mockUUID]: {
            group: ExperimentGroup.Treatment,
            name: mockExperimentMetadata.name,
            type: 'device_id',
            uuid: mockUUID,
            version: 1,
          },
        });
      });

      it('applies overrides when debug is enabled', () => {
        expect(
          getExperimentGroupsForUser({
            bucket: 'a',
            debug: true,
            experimentOverrides: `${mockUUID}=${ExperimentGroup.Control}`,
            experimentSlots: {
              0: mockUUID,
              1: undefined,
              2: undefined,
            },
            experiments: {
              [mockUUID]: mockExperimentMetadata,
            },
          }),
        ).toEqual({
          [mockUUID]: {
            group: ExperimentGroup.Control,
            name: mockExperimentMetadata.name,
            type: 'device_id',
            uuid: mockUUID,
            version: 1,
          },
        });
      });
    });
  });
});
