import type { ParsedUrlQuery } from 'querystring';
import { Enum } from 'tachyon-utils-ts';
import { ExperimentGroup } from '../../experimentInfo';
import { printInvalidOverridesWarning } from '../warnings';

export type ExperimentOverrides = {
  [uuid: string]: ExperimentGroup;
};

/**
 * Parse the query parameter "experiment_overrides" to explicitly set experiment
 * groups.
 *
 * Exported for testing.
 *
 * @param experimentOverridesQuery The value of the "experiment_overrides"
 *   query parameter.
 *   (e.g. "12345678-1234-1234-1234-123456789012=treatment,uuid2=control")
 */
export function parseExperimentOverrides(
  experimentOverridesQuery: ParsedUrlQuery[string],
  invalidHandler?: (override: string) => void,
): ExperimentOverrides {
  if (typeof experimentOverridesQuery === 'string') {
    experimentOverridesQuery = experimentOverridesQuery.split(',');
  } else if (!experimentOverridesQuery) {
    experimentOverridesQuery = [];
  }

  return experimentOverridesQuery.reduce<ExperimentOverrides>(
    (acc, override) => {
      const [uuid, group] = override.split('=');
      const experimentGroup = Enum.convertValueFromExternal(
        ExperimentGroup,
        group,
      );

      if (uuid && experimentGroup) {
        acc[uuid] = experimentGroup;
      } else {
        invalidHandler?.(override);
      }

      return acc;
    },
    {},
  );
}

export function parseExperimentOverridesWithWarnings(
  experimentOverridesQuery: ParsedUrlQuery[string],
): ExperimentOverrides {
  const invalidOverrides: string[] = [];
  const overrides = parseExperimentOverrides(
    experimentOverridesQuery,
    (override) => {
      invalidOverrides.push(override);
    },
  );

  if (experimentOverridesQuery && invalidOverrides.length) {
    printInvalidOverridesWarning(invalidOverrides, experimentOverridesQuery);
  }
  return overrides;
}
