import type { ParsedUrlQuery } from 'querystring';
import { ExperimentGroup } from '../../experimentInfo';
import type { ExperimentOverrides } from '.';
import { parseExperimentOverrides } from '.';

describe(parseExperimentOverrides, () => {
  type ExperimentOverrideTestCase = {
    readonly expected: ExperimentOverrides;
    readonly query: ParsedUrlQuery[string];
  };

  const mockUUID = 'foo';

  const TEST_CASES: ExperimentOverrideTestCase[] = [
    {
      expected: {
        [mockUUID.toUpperCase()]: ExperimentGroup.Treatment,
      },
      query: 'FOO=treatment',
    },
    {
      expected: {
        [mockUUID]: ExperimentGroup.Treatment,
      },
      query: 'foo=treatment',
    },
    {
      expected: {
        [mockUUID]: ExperimentGroup.Treatment,
      },
      query: 'foo=TREATMENT',
    },
    {
      expected: {
        [mockUUID]: ExperimentGroup.Treatment,
      },
      query: 'not_a_uuid=whatever,foo=TREATMENT',
    },
    {
      expected: {
        [mockUUID]: ExperimentGroup.Treatment,
      },
      query: 'not_a_uuid=whatever,foo=TREATMENT,also_not_a_uuid=whatever,',
    },
    {
      expected: {
        [mockUUID]: ExperimentGroup.Treatment,
      },
      query: 'foo=TREATMENT,also_not_a_uuid=whatever',
    },
    {
      expected: {},
      query: '',
    },
    {
      expected: {},
      query: 'foo===',
    },
    {
      expected: {},
      query: 'foo=not_a_group',
    },
    {
      expected: {
        [mockUUID]: ExperimentGroup.Treatment,
      },
      query: 'foo=TREATMENT,=whatever',
    },
    {
      expected: {
        [mockUUID]: ExperimentGroup.Treatment,
      },
      query: 'foo=TREATMENT,also_not_a_uuid=',
    },
    {
      expected: {
        [mockUUID]: ExperimentGroup.Treatment,
      },
      query: ',,,also_not_a_uuid=,,foo=TREATMENT,',
    },
    {
      expected: {},
      query: ',,,also_not_a_uuid=,,foo==TREATMENT,',
    },
    {
      expected: {},
      query: ',,,==&also_not_a_uuid=,,foo==TREATMENT,',
    },
    {
      expected: {
        [mockUUID]: ExperimentGroup.Treatment,
      },
      query: ',,,==&also_not_a_uuid=,,foo=TREATMENT,',
    },
    {
      expected: {},
      query: undefined,
    },
  ];
  TEST_CASES.forEach(({ expected, query }) => {
    it(`parses ${query}`, () => {
      expect(parseExperimentOverrides(query)).toEqual(expected);
    });
  });
});
