import type {
  ExperimentMetadata,
  WeightedExperimentGroup,
} from '../../experimentInfo';
import { ExperimentGroup } from '../../experimentInfo';
import {
  printAllTreatmentExperimentWarning,
  printExperimentNotFoundWarning,
  printInvalidSplitWarning,
  printNotTreatmentAndControlWarning,
} from '../warnings';

/**
 * A value grouped with (ideally) the percentage detailing how often it should
 * be applied. In reality, people can use numeric bases other than 100 for their
 * splits (e.g. 34 & 453).
 *
 * Presently, only 0/100, 50/50 and 100/0 splits are supported.
 * Arbitraty splits (such as 34/453) will be applied as 50/50.
 */
export type MinixperimentGroup = {
  /**
   * The stringy value for this group.
   */
  readonly value: string;
  /**
   * The likelihood a candidate is put in this group.
   */
  readonly weight: number;
};

/**
 * An enumeration describing the type of experiment in Minixperiment.
 */
export enum MinixperimentType {
  /**
   * An experiment that is sticky to the "device" (really browser).
   */
  DeviceId = 1,
  /**
   * An experiment that is sticky to a user across any number of browsers.
   */
  UserId,
  /**
   * An experiment that is sticky to a channel regardless of device or user.
   */
  ChannelId,
}

/**
 * An experiment currently defined in Minixperiment.
 */
export type Minixperiment = {
  /**
   * The different treatment groups for an experiment.
   */
  readonly groups: MinixperimentGroup[];
  /**
   * The human-readable name for an experiment.
   */
  readonly name: string;
  /**
   * The type of experiment.
   */
  readonly t: MinixperimentType;
  /**
   * The version of experiments.json experiment was updated last.
   */
  readonly v: number;
};

/**
 * A mapping of experiments in Minixperiment keyed on the experiment UUID.
 */
export type Minixperiments = {
  [experimentUUID: string]: Minixperiment | undefined;
};

const allControlGroups: WeightedExperimentGroup[] = [
  { value: ExperimentGroup.Control, weight: 100 },
  { value: ExperimentGroup.Treatment, weight: 0 },
];

function isValidSplit<Group extends MinixperimentGroup>(
  groups: Group[],
): groups is Array<Group & Pick<WeightedExperimentGroup, 'weight'>> {
  return (
    groups.every((group) => [0, 50, 100].includes(group.weight)) &&
    groups[0].weight + groups[1].weight === 100
  );
}

function isTreatmentAndControl<Group extends MinixperimentGroup>(
  groups: Group[],
): groups is Array<Group & Pick<WeightedExperimentGroup, 'value'>> {
  const set = new Set(groups.map((group) => group.value));
  return (
    set.size === 2 &&
    set.has(ExperimentGroup.Control) &&
    set.has(ExperimentGroup.Treatment)
  );
}

type MinixperimentProccessError =
  | 'INVALID_SPLIT'
  | 'NOT_FOUND'
  | 'NOT_TREATMENT_AND_CONTROL';

export function processMinixperiments(
  minixperiments: Minixperiments,
  invalidHandler?: (uuid: string, error: MinixperimentProccessError) => void,
): ExperimentMetadata[] {
  return Object.entries(minixperiments).reduce<ExperimentMetadata[]>(
    (acc, [uuid, minixperiment]) => {
      if (minixperiment === undefined) {
        invalidHandler?.(uuid, 'NOT_FOUND');
        return acc;
      }

      const { groups, name, v: version } = minixperiment;
      const experiment = { name, type: 'device_id' as const, uuid, version };

      if (!isTreatmentAndControl(groups)) {
        invalidHandler?.(uuid, 'NOT_TREATMENT_AND_CONTROL');
        acc.push({
          ...experiment,
          groups: allControlGroups,
        });
      } else if (!isValidSplit(groups)) {
        invalidHandler?.(uuid, 'INVALID_SPLIT');
        acc.push({
          ...experiment,
          groups: allControlGroups,
        });
      } else {
        acc.push({
          ...experiment,
          groups,
        });
      }

      return acc;
    },
    [],
  );
}

/**
 * Same as processMinixperimentResponse but prints an obvious console warning
 * for detected issues. Also flags experiments that should be removed:
 *
 * When treatments are set to 100%, this inverts the weighting so control is 100%
 * and treatment is 0%.
 *
 * This should only be used in dev mode.
 */
export function processMinixperimentsWithWarnings(
  minixperiments: Partial<Minixperiments>,
): ExperimentMetadata[] {
  const experimentsForApp = processMinixperiments(
    minixperiments,
    (uuid, error) => {
      switch (error) {
        case 'NOT_FOUND': {
          printExperimentNotFoundWarning(uuid);
          break;
        }
        case 'NOT_TREATMENT_AND_CONTROL': {
          printNotTreatmentAndControlWarning(uuid);
          break;
        }
        case 'INVALID_SPLIT': {
          printInvalidSplitWarning(uuid);
          break;
        }
      }
    },
  );

  experimentsForApp.forEach((experiment) => {
    const isAllTreatment = experiment.groups.some(
      (group) =>
        group.value === ExperimentGroup.Treatment && group.weight === 100,
    );

    if (isAllTreatment) {
      // override readonly for dev-only transform
      (experiment as any).groups = allControlGroups;
      printAllTreatmentExperimentWarning(experiment.name);
    }
  });

  return experimentsForApp;
}
