/* eslint-disable no-console */
import type { ExperimentMetadata } from '../../experimentInfo';
import { ExperimentGroup } from '../../experimentInfo';
import type { Minixperiments } from '.';
import {
  MinixperimentType,
  processMinixperiments,
  processMinixperimentsWithWarnings,
} from '.';

jest.spyOn(global.console, 'warn').mockImplementation(() => null);

describe(processMinixperiments, () => {
  beforeEach(() => {
    (console.warn as jest.Mock).mockReset();
  });

  type ExperimentTestCase = {
    data: Partial<Minixperiments>;
    expected: ExperimentMetadata[];
    name: string;
  };

  const mockUUID1 = '1';
  const mockUUID2 = '2';
  const mockUUID3 = '3';
  const mockUUID4 = '4';
  const mockUUID5 = '5';

  const testCases: ExperimentTestCase[] = [
    {
      data: {
        [mockUUID1]: {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 50,
            },
            {
              value: ExperimentGroup.Control,
              weight: 50,
            },
          ],
          name: 'mobile_web_experiment_1',
          t: MinixperimentType.DeviceId,
          v: 0,
        },
        [mockUUID2]: {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
          ],
          name: 'mobile_web_experiment_2',
          t: MinixperimentType.DeviceId,
          v: 1,
        },
      },
      expected: [
        {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 50,
            },
            {
              value: ExperimentGroup.Control,
              weight: 50,
            },
          ],
          name: 'mobile_web_experiment_1',
          type: 'device_id',
          uuid: mockUUID1,
          version: 0,
        },
        {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
          ],
          name: 'mobile_web_experiment_2',
          type: 'device_id',
          uuid: mockUUID2,
          version: 1,
        },
      ],
      name: 'returns multi-item array when response has multiple relevant experiments',
    },
    {
      data: {
        [mockUUID1]: undefined,
      },
      expected: [],
      name: 'returns empty array when input experiments are missing metadata',
    },
    {
      data: {
        [mockUUID1]: {
          groups: [],
          name: 'mobile_web_experiment_1',
          t: MinixperimentType.DeviceId,
          v: 0,
        },
        [mockUUID2]: {
          groups: [{ value: ExperimentGroup.Control, weight: 0 }],
          name: 'mobile_web_experiment_2',
          t: MinixperimentType.DeviceId,
          v: 0,
        },
        [mockUUID3]: {
          groups: [{ value: ExperimentGroup.Treatment, weight: 0 }],
          name: 'mobile_web_experiment_3',
          t: MinixperimentType.DeviceId,
          v: 0,
        },
        [mockUUID4]: {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
            {
              value: 'other thing',
              weight: 0,
            },
          ],
          name: 'mobile_web_experiment_4',
          t: MinixperimentType.DeviceId,
          v: 0,
        },
        [mockUUID5]: {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 100,
            },
          ],
          name: 'mobile_web_experiment_5',
          t: MinixperimentType.DeviceId,
          v: 0,
        },
      },
      expected: [
        {
          groups: [
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
          ],
          name: 'mobile_web_experiment_1',
          type: 'device_id',
          uuid: mockUUID1,
          version: 0,
        },
        {
          groups: [
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
          ],
          name: 'mobile_web_experiment_2',
          type: 'device_id',
          uuid: mockUUID2,
          version: 0,
        },
        {
          groups: [
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
          ],
          name: 'mobile_web_experiment_3',
          type: 'device_id',
          uuid: mockUUID3,
          version: 0,
        },
        {
          groups: [
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
          ],
          name: 'mobile_web_experiment_4',
          type: 'device_id',
          uuid: mockUUID4,
          version: 0,
        },
        {
          groups: [
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
          ],
          name: 'mobile_web_experiment_5',
          type: 'device_id',
          uuid: mockUUID5,
          version: 0,
        },
      ],
      name: 'returns control only groups when experiments have invalid groups',
    },
    {
      data: {
        [mockUUID1]: {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
            {
              value: ExperimentGroup.Control,
              weight: 10,
            },
          ],
          name: 'mobile_web_experiment_1',
          t: MinixperimentType.DeviceId,
          v: 0,
        },
        [mockUUID2]: {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 50,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 100,
            },
          ],
          name: 'mobile_web_experiment_2',
          t: MinixperimentType.DeviceId,
          v: 0,
        },
        [mockUUID3]: {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 50,
            },
          ],
          name: 'mobile_web_experiment_3',
          t: MinixperimentType.DeviceId,
          v: 0,
        },
        [mockUUID4]: {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 40,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 60,
            },
          ],
          name: 'mobile_web_experiment_4',
          t: MinixperimentType.DeviceId,
          v: 0,
        },
      },
      expected: [
        {
          groups: [
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
          ],
          name: 'mobile_web_experiment_1',
          type: 'device_id',
          uuid: mockUUID1,
          version: 0,
        },
        {
          groups: [
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
          ],
          name: 'mobile_web_experiment_2',
          type: 'device_id',
          uuid: mockUUID2,
          version: 0,
        },
        {
          groups: [
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
          ],
          name: 'mobile_web_experiment_3',
          type: 'device_id',
          uuid: mockUUID3,
          version: 0,
        },
        {
          groups: [
            {
              value: ExperimentGroup.Control,
              weight: 100,
            },
            {
              value: ExperimentGroup.Treatment,
              weight: 0,
            },
          ],
          name: 'mobile_web_experiment_4',
          type: 'device_id',
          uuid: mockUUID4,
          version: 0,
        },
      ],
      name: 'returns control only groups when experiments have invalid weights',
    },
  ];

  testCases.forEach(({ data, expected, name }) => {
    it(name, () => {
      expect(processMinixperiments(data)).toEqual(expected);
    });
  });

  describe(processMinixperimentsWithWarnings, () => {
    it('displays warning when given a relevant 100% treatment group', () => {
      const data = {
        [mockUUID1]: {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 50,
            },
            {
              value: ExperimentGroup.Control,
              weight: 50,
            },
          ],
          name: 'mobile_web_experiment_1',
          t: MinixperimentType.DeviceId,
          v: 0,
        },
        [mockUUID2]: {
          groups: [
            {
              value: ExperimentGroup.Treatment,
              weight: 100,
            },
            {
              value: ExperimentGroup.Control,
              weight: 0,
            },
          ],
          name: 'mobile_web_experiment_2',
          t: MinixperimentType.DeviceId,
          v: 1,
        },
      };
      const result = processMinixperimentsWithWarnings(data);
      const preservedWeightExperiment = result[0];
      expect(preservedWeightExperiment.groups).toEqual([
        {
          value: ExperimentGroup.Treatment,
          weight: 50,
        },
        {
          value: ExperimentGroup.Control,
          weight: 50,
        },
      ]);

      const allTreatmentExperiment = result[1];
      expect(allTreatmentExperiment.groups).toEqual([
        {
          value: ExperimentGroup.Control,
          weight: 100,
        },
        {
          value: ExperimentGroup.Treatment,
          weight: 0,
        },
      ]);

      expect(console.warn).toHaveBeenCalledTimes(1);
      expect(console.warn).toHaveBeenCalledWith(
        expect.stringContaining(
          'The following experiment is at 100% treatment',
        ),
      );
      expect(console.warn).toHaveBeenCalledWith(
        expect.stringContaining(mockUUID2),
      );
    });
  });
});
