import type { IntlData } from 'tachyon-intl';
import { SUPPORTED_LOCALE_DATA } from 'tachyon-intl';
import type { BaseLocale } from 'twitch-intl';
import type { LocaleMessageData } from './TachyonIntl';
import { loadCLDRForLocale } from './cldrLoader';
import type { ParseMessageDataOpts } from './parseMessageData';
import { parseMessageData } from './parseMessageData';
import { loadFormatDataHydrationScriptsForLocale } from './polyfillScriptLoader';

/**
 * A array of IntlData objects. This in-module state is designed to be long-lived
 * (same datat used for many server requests) which is why we explicitly make
 * this type read-only.
 */
export type IntlDataCache = ReadonlyArray<Readonly<IntlData>>;

/**
 * Local cache designed to live across multiple server requests.
 */
let intlDataCache: IntlDataCache = [];

/**
 * Does all of the necessary work to prepare a locale for consumption by TachyonIntl
 * including setting any associated message data and loading corresponding CLDR rules.
 */
export async function prepareIntlDataCacheValue(
  baseLocale: BaseLocale,
  messageData?: LocaleMessageData,
): Promise<IntlData> {
  const formatDataHydrationScripts =
    await loadFormatDataHydrationScriptsForLocale(baseLocale.languageCode);

  return {
    formatDataHydrationScripts,
    locales: [
      {
        ...baseLocale,
        data: {
          formatData: loadCLDRForLocale(baseLocale.locale),
          locale: baseLocale.locale,
          messages: messageData ? messageData[baseLocale.languageCode] : {},
        },
      },
    ],
    preferredLanguageTags: [baseLocale.languageCode],
  };
}

export type PrepareIntlDataCacheOpts = ParseMessageDataOpts;

/**
 * Populates the locale data cache using the messages for each locale provided
 * by the application, polyfills the Intl on the server and loads CLDR rules for
 * each locale.
 *
 * @param messageData Translated strings for an application mapped to each locale.
 */
export async function prepareIntlDataCache(
  opts: PrepareIntlDataCacheOpts = {},
): Promise<void> {
  // We polyfill this since Node's Intl implementation isn't entirely complete (at least with
  // regards to CLDR rules) so you end up with circumstances where internationalized strings
  // will render differently on the server, and then immediately change when JavaScript on the client
  // reclaims the DOM.
  //
  // Note: This polyfill is applied inside of this function to avoid side-effects that would cause
  // this to also be applied client-side since this function is exported along side code for the client.
  global.Intl = require('intl');

  const messageData = parseMessageData(opts);

  intlDataCache = await Promise.all(
    SUPPORTED_LOCALE_DATA.map((supportedLocale) =>
      prepareIntlDataCacheValue(supportedLocale, messageData),
    ),
  );
}

export function getIntlDataCache(): IntlDataCache {
  return intlDataCache;
}

/**
 * Only used for testing purposes.
 */
export function clearIntlDataCache(): void {
  intlDataCache = [];
}
