import { join, resolve } from 'path';
import { cwd } from 'process';
import { SUPPORTED_LOCALE_DATA } from 'tachyon-intl';
import { logger } from 'tachyon-logger';
import type { LocaleMessages } from 'twitch-intl';
import type { LocaleMessageData } from '../TachyonIntl';
import { dynamicRequire } from '../utils';

export const applicationErrorMessage =
  'Application has no entries for supported language code:';
export const packageErrorMessage =
  'Package with custom translations has no entries for supported lanugage code:';
export const overwriteErrorMessage =
  'Package tried to overwrite existing translation key:';

export interface ParseMessageDataOpts {
  appDir?: string;
  includePackages?: string[];
}

/**
 * Read all of the translated messages into an object as preparation to locale
 * caching.
 *
 * @param appDir The directory containing the app's `messages` directory
 * @param includePackages Any packages with translations to be included. Those
 * packages must have a top-level `messages` directory to work.
 */
export function parseMessageData(
  { appDir = cwd(), includePackages = [] }: ParseMessageDataOpts = {
    appDir: cwd(),
    includePackages: [],
  },
): LocaleMessageData {
  return SUPPORTED_LOCALE_DATA.reduce<LocaleMessageData>(
    (acc, { default: isDefault, languageCode }) => {
      let messages: LocaleMessages = {};

      // the default language corresponds to the strings in the app code, so
      // sending them in the intl payload is just duplication since intl will
      // fallback to in the in-app definitions
      if (isDefault) {
        acc[languageCode] = messages;

        return acc;
      }

      const appMessagesPath = resolve(
        appDir,
        'messages',
        languageCode,
        'messages.json',
      );

      try {
        // can't use fs.existsSync here because webpack is weird with fs
        messages = dynamicRequire(appMessagesPath);
      } catch {
        logger.warn({
          category: 'parseMessageData',
          message: `${applicationErrorMessage} ${languageCode}`,
          package: 'tachyon-intl-server',
        });
      }

      includePackages.forEach((pkg) => {
        const packageMessagesPath = join(
          pkg,
          'messages',
          languageCode,
          'messages.json',
        );

        let packageMessages: LocaleMessages | undefined;
        try {
          packageMessages = dynamicRequire(packageMessagesPath);
        } catch {
          logger.warn({
            category: 'parseMessageData',
            message: `${packageErrorMessage} ${pkg} - ${languageCode}`,
            package: 'tachyon-intl-server',
          });
        }

        if (packageMessages) {
          Object.entries(packageMessages).forEach(([key, message]) => {
            if (messages[key]) {
              // don't overwrite messages, log warning instead
              logger.warn({
                category: 'parseMessageData',
                message: `${overwriteErrorMessage} ${pkg} - ${key} - ${languageCode}`,
                package: 'tachyon-intl-server',
              });

              return;
            }

            messages[key] = message;
          });
        }
      });

      acc[languageCode] = messages;

      return acc;
    },
    {},
  );
}
