import { resolve } from 'path';
import { logger } from 'tachyon-logger';
import type { LocaleMessageData } from '../TachyonIntl';
import {
  applicationErrorMessage,
  overwriteErrorMessage,
  parseMessageData,
} from '.';

jest.mock('tachyon-logger', () => ({ logger: { warn: jest.fn() } }));

describe('parseMessageData', () => {
  let messageData: LocaleMessageData;
  beforeEach(() => {
    (logger.warn as jest.Mock).mockReset();
  });

  const appDir = resolve(__dirname, '..', '..');

  describe("for application's translations", () => {
    beforeEach(() => {
      messageData = parseMessageData({ appDir });
    });

    it('does not load translations for default langauge', () => {
      expect(typeof messageData.en).toBe('object');
      expect(Object.keys(messageData.en)).toHaveLength(0);
      expect(messageData.en['[NameSpace] Test Phrase']).toBeUndefined();
    });

    it('finds translations for supported langauges', () => {
      expect(typeof messageData.es).toBe('object');
      expect(Object.keys(messageData.es).length).toBeGreaterThan(0);
      expect(messageData.es['[NameSpace] Test Phrase']).toEqual(
        'Frase de prueba',
      );
    });

    it('warns without error when no translations for supported language', () => {
      expect(messageData.fi).toEqual({});
      expect(logger.warn).toHaveBeenCalledWith({
        category: 'parseMessageData',
        message: expect.stringContaining(applicationErrorMessage),
        package: 'tachyon-intl-server',
      });
    });
  });

  describe('with specified app directory', () => {
    it('finds translation', () => {
      messageData = parseMessageData({ appDir });
      expect(typeof messageData.es).toBe('object');
      expect(messageData.es['[NameSpace] Test Phrase']).toBeDefined();
    });
  });

  // leans on tachyon-more-ui being in the monorepo
  describe('with package custom translations', () => {
    beforeEach(() => {
      messageData = parseMessageData({
        appDir,
        includePackages: ['tachyon-more-ui'],
      });
    });

    it('contains content from a package with its own translations', () => {
      expect(messageData.fr['[Status] {name} is offline']).toBeDefined();
    });

    it('warns when a package tries to overwrite an already defined translation key', () => {
      expect(logger.warn).toHaveBeenCalledWith({
        category: 'parseMessageData',
        message: expect.stringContaining(overwriteErrorMessage),
        package: 'tachyon-intl-server',
      });
      expect(messageData.es['[Status] {name} is offline']).toEqual(
        'Test for message collisions',
      );
    });
  });
});
