import { readFileSync } from 'fs';
import { minify } from 'terser';
import { dynamicRequire } from '../utils';

enum ScriptType {
  Message = 'message',
  Relative = 'relative',
}

export async function prepareScript(
  rawScript: string,
  type: ScriptType,
): Promise<string> {
  // if minify fails, we fall back to old carriage-return stripping
  switch (type) {
    case ScriptType.Message:
      const messagePartialMinify = `var ald = IntlMessageFormat.__addLocaleData;${rawScript.replace(
        /IntlMessageFormat\.__addLocaleData/g,
        'ald',
      )}`;

      const { code: minifiedMessage } = await minify(messagePartialMinify, {
        mangle: { toplevel: true },
      });

      return minifiedMessage || messagePartialMinify.replace(/\n/, '');
    case ScriptType.Relative:
      const relativePartialMinify = `var ald = IntlRelativeFormat.__addLocaleData;${rawScript.replace(
        /IntlRelativeFormat\.__addLocaleData/g,
        'ald',
      )}`;

      const { code: minifiedRelative } = await minify(relativePartialMinify, {
        mangle: { toplevel: true },
      });

      return minifiedRelative || relativePartialMinify.replace(/\n/, '');
    default:
      return type;
  }
}

/**
 * Loads polyfills for various Intl Browser APIs that are required for a given
 * Locale to work properly.
 */
export async function loadFormatDataHydrationScriptsForLocale(
  languageCode: string,
): Promise<string[]> {
  const customLocaleKey = languageCode.split('-')[0];

  const messageFormatKey = customLocaleKey;

  let relativeFormatKey = customLocaleKey;
  // some macrolanguages get overriden to specific individual languages
  if (relativeFormatKey === 'no') {
    relativeFormatKey = 'nb';
  }

  // These polyfills are raw JS statements (no exports) and thus automatically
  // patch their respective libraries via global reference immediately upon
  // parsing/execution. Since we want to programmatically handle locale config,
  // we get the raw text of the files in order to send to the client to be
  // eval-ed as part of client re-hydration. We also do a normal require
  // so that we properly hydrate the server environment.
  const messageFormatScript = `intl-messageformat/dist/locale-data/${messageFormatKey}.js`;
  dynamicRequire(messageFormatScript);
  const messageFormatScriptText = readFileSync(
    dynamicRequire.resolve(messageFormatScript),
    'utf8',
  );

  const relativeFormatScript = `intl-relativeformat/dist/locale-data/${relativeFormatKey}.js`;
  dynamicRequire(relativeFormatScript);
  const relativeFormatScriptText = readFileSync(
    dynamicRequire.resolve(relativeFormatScript),
    'utf8',
  );

  return [
    await prepareScript(messageFormatScriptText, ScriptType.Message),
    await prepareScript(relativeFormatScriptText, ScriptType.Relative),
  ];
}
