// eslint-disable-next-line tachyon/no-react-import
import React from 'react';
import type { IntlData } from 'tachyon-intl';
import { logger } from 'tachyon-logger';
import { TwitchIntl } from 'twitch-intl';
import { getIntlDataCache } from '../intlDataCache';

/**
 * Returns the most appropriate locale bundle, as selected by TwitchIntl based
 * on the user's language preferences.
 *
 * @throws If no locale is successfully loaded, or somehow, one was loaded that was not expected.
 * @param preferredLanguageTags A list of https://tools.ietf.org/html/bcp47 language tags. (es-ES, es, en-US, en, etc).
 * @return an object with both locale messages and raw scripts to execute on the client
 */
export function selectIntlData(preferredLanguageTags: string[]): IntlData {
  // Defer the selection of locales to TwitchIntl since it will deal with all
  // of the fall-through cases for us.
  const intlDataCache = getIntlDataCache();
  if (intlDataCache.length === 0) {
    const errorMessage =
      'No locale bundles found, ensure that "prepareIntlDataCache" is being called';
    logger.error({
      category: 'selectIntlData',
      message: errorMessage,
      package: 'tachyon-intl-server',
    });

    throw new Error(errorMessage);
  }

  const allLocales = intlDataCache.map((bundle) => bundle.locales[0]);
  const intlSelector = new TwitchIntl(allLocales, React);

  const selectedLocaleData = intlSelector.loadLocaleSync(preferredLanguageTags);

  if (!selectedLocaleData) {
    const errorMessage =
      'TachyonIntl failed to select a locale from those that were provided.';
    logger.error({
      category: 'selectLocaleBundle',
      message: errorMessage,
      package: 'tachyon-intl-server',
    });

    throw new Error(errorMessage);
  }

  // Convert back to our Tachyon Locale format
  const selectedIntlData = intlDataCache.find(
    ({ locales }) => locales[0].locale === selectedLocaleData.locale,
  );

  if (!selectedIntlData) {
    const errorMessage = `No Locale match found for selected locale ${selectedLocaleData.locale}.`;
    logger.error({
      category: 'selectIntlData',
      message: errorMessage,
      package: 'tachyon-intl-server',
    });

    throw new Error(errorMessage);
  }

  return selectedIntlData;
}
