// eslint-disable-next-line tachyon/no-react-import
import React, { createContext } from 'react';
import type { FC } from 'react';
import { useConst } from 'tachyon-utils-react';
import { isBrowser } from 'tachyon-utils-stdlib';
import type { StaticLocale } from 'twitch-intl';
import { TwitchIntl } from 'twitch-intl';
import type { RequireExactlyOne } from 'type-fest';
import { DefaultTwitchIntl } from '../DefaultTwitchIntl';

/**
 * The context provided by IntlRoot.
 */
export type IntlContext = {
  /**
   * An instance of TachyonIntl(TwitchIntl).
   */
  intl: TwitchIntl;
};

export const intlContext = createContext<IntlContext>({
  intl: DefaultTwitchIntl,
});

export type IntlData = {
  /**
   * List of raw JS scripts as strings used to populate IntlMessageFormat and
   * IntlRelativeFormat locale data rules. These hydration scripts will
   * automatically be evaluated at component initialization in a client-side
   * context.
   */
  formatDataHydrationScripts?: string[];
  /**
   * Information about locales including the messages that are defined in the
   * app. This can be an array of a single StaticLocale targeted to the current
   * user or an array of all of the possible StaticLocales.
   */
  locales: StaticLocale[];
  /**
   * A list of https://tools.ietf.org/html/bcp47 language tags (es-ES, es,
   * en-US, en, etc) relevant to the current user, in order of preference.
   */
  preferredLanguageTags: string[];
};

export type TachyonIntlRootProps = RequireExactlyOne<
  {
    /**
     * Data necessary to construct and initialize a TwitchIntl instance
     */
    data: IntlData;
    /**
     * TwitchIntl instance that will be re-used within tachyon-intl
     */
    intl: TwitchIntl;
  },
  'data' | 'intl'
>;

/**
 * Manages a React Context for consuming locale data via TwitchIntl. You'll
 * notice this component is server aware to make it easier for SSR consumers
 * to utilize this component. TwitchIntl is created in the constructor as that
 * makes it easy for SSR consumers to serialize and rehydrate this component.
 */
export const TachyonIntlRoot: FC<TachyonIntlRootProps> = (props) => {
  // don't destructure so TS can track the RequireExactlyOne semantics
  const ctx = useConst<IntlContext>(() => {
    if (props.intl) {
      return { intl: props.intl };
    }

    const { formatDataHydrationScripts, locales, preferredLanguageTags } =
      props.data;

    const intl = new TwitchIntl(locales, React);
    intl.loadLocaleSync(preferredLanguageTags);

    // The hydration scripts eval'd below are dependent on libraries being
    // available in a client-side context which happens as a side-effect of importing
    // TwitchIntl. We do this in the constructor to ensure that the first render
    // for all Intl consumers has this formatting data.
    if (isBrowser() && formatDataHydrationScripts) {
      const polyfillScriptContent = `${formatDataHydrationScripts.join(';')};`;
      // eslint-disable-next-line no-eval
      eval(polyfillScriptContent);
    }

    return { intl };
  });

  return <intlContext.Provider children={props.children} value={ctx} />;
};

TachyonIntlRoot.displayName = 'TachyonIntlRoot';
