import type { BaseLocale } from 'twitch-intl';

// IMPORTANT INFO ABOUT THIS DATA STRUCTURE:
// 1) Order matters. The first locale in this list for a given language will be
// used as a fallback if a user asks for a similar regional variant that we do not support.
//
// 2) Make sure that there is at least one locale in the list with the "default"
// field set to true. This locale will be used as a last resort if we do not
// at all support any of the users preferences or anything remotely similar.

/**
 * The list of locale objects used for configuring an application that uses
 * the twitch-intl package.
 */
export const SUPPORTED_LOCALE_DATA: ReadonlyArray<BaseLocale> = [
  {
    languageCode: 'bg',
    locale: 'bg-BG',
    name: 'Български',
  },
  {
    cldrLocale: 'cs-cz',
    languageCode: 'cs',
    locale: 'cs-CZ',
    name: 'Čeština',
  },
  {
    languageCode: 'da',
    locale: 'da-DK',
    name: 'Dansk',
  },
  {
    languageCode: 'de',
    locale: 'de-DE',
    name: 'Deutsch',
  },
  {
    languageCode: 'el',
    locale: 'el-GR',
    name: 'Ελληνικά',
  },
  {
    default: true,
    languageCode: 'en',
    locale: 'en-US',
    name: 'English',
  },
  {
    intlMessageFormatKey: 'en',
    languageCode: 'en-gb',
    locale: 'en-GB',
    name: 'English - UK',
  },
  {
    languageCode: 'es',
    locale: 'es-ES',
    name: 'Español - España',
  },
  {
    intlMessageFormatKey: 'es',
    languageCode: 'es-mx',
    locale: 'es-MX',
    name: 'Español - Latinoamérica',
  },
  {
    languageCode: 'fi',
    locale: 'fi-FI',
    name: 'Suomi',
  },
  {
    languageCode: 'fr',
    locale: 'fr-FR',
    name: 'Français',
  },
  {
    languageCode: 'hu',
    locale: 'hu-HU',
    name: 'Magyar',
  },
  {
    languageCode: 'it',
    locale: 'it-IT',
    name: 'Italiano',
  },
  {
    languageCode: 'ja',
    locale: 'ja-JP',
    name: '日本語',
  },
  {
    languageCode: 'ko',
    locale: 'ko-KR',
    name: '한국어',
  },
  {
    languageCode: 'nl',
    locale: 'nl-NL',
    name: 'Nederlands',
  },
  {
    cldrLocale: 'nb',
    languageCode: 'no',
    locale: 'no-NO',
    name: 'Norsk',
  },
  {
    languageCode: 'pl',
    locale: 'pl-PL',
    name: 'Polski',
  },
  {
    cldrLocale: 'pt-pt',
    intlMessageFormatKey: 'pt-pt',
    languageCode: 'pt',
    locale: 'pt-PT',
    name: 'Português',
  },
  {
    cldrLocale: 'pt',
    intlMessageFormatKey: 'pt',
    languageCode: 'pt-br',
    locale: 'pt-BR',
    name: 'Português - Brasil',
  },
  {
    languageCode: 'ro',
    locale: 'ro-RO',
    name: 'Română',
  },
  {
    languageCode: 'ru',
    locale: 'ru-RU',
    name: 'Русский',
  },
  {
    languageCode: 'sk',
    locale: 'sk-SK',
    name: 'Slovenčina',
  },
  {
    languageCode: 'sv',
    locale: 'sv-SE',
    name: 'Svenska',
  },
  {
    languageCode: 'th',
    locale: 'th-TH',
    name: 'ภาษาไทย',
  },
  {
    languageCode: 'tr',
    locale: 'tr-TR',
    name: 'Türkçe',
  },
  {
    languageCode: 'vi',
    locale: 'vi-VN',
    name: 'Tiếng Việt',
  },
  {
    cldrLocale: 'zh-hans',
    intlMessageFormatKey: 'zh',
    languageCode: 'zh-cn',
    locale: 'zh-CN',
    name: '中文 简体',
  },
  {
    cldrLocale: 'zh-hant',
    intlMessageFormatKey: 'zh-hant',
    languageCode: 'zh-tw',
    locale: 'zh-TW',
    name: '中文 繁體',
  },
];

// istanbul ignore next: trivial
/**
 * The list of locales that are supported by Twitch. This is a convenience
 * structure computed from SUPPORTED_LOCALE_DATA to prevent the need for
 * duplication of this logic in multiple places.
 */
export const SUPPORTED_LOCALES: ReadonlyArray<string> =
  SUPPORTED_LOCALE_DATA.reduce<string[]>((agg, val) => {
    agg.push(val.locale);
    return agg;
  }, []);

// istanbul ignore next: trivial
/**
 * The list of language codes that are supported by Twitch. This is a convenience
 * structure computed from SUPPORTED_LOCALE_DATA to prevent the need for
 * duplication of this logic in multiple places.
 */
export const SUPPORTED_LANGUAGE_CODES: ReadonlyArray<string> =
  SUPPORTED_LOCALE_DATA.reduce<string[]>((agg, val) => {
    agg.push(val.languageCode);
    return agg;
  }, []);
