import type { ComponentType, FC } from 'react';
import { getComponentName } from 'tachyon-utils-react';
import type { IntlContext } from '../TachyonIntlRoot';
import { useIntl } from '../useIntl';

export type IntlProps = IntlContext;

/**
 * Wrapping a component in withIntl will provide that component
 * with the `intl` prop. This prop is an instance of TachyonIntl, providing
 * methods such as formatMessage and formatNumber. These methods must be
 * destructured for proper functionality; formatMessage will work but
 * intl.formatMessage will not get picked up by the current tooling.
 *
 * @param Component A component which needs intl.
 */
export function withIntl<P extends IntlProps>(
  Comp: ComponentType<P>,
): FC<Omit<P, keyof IntlProps>> {
  const Intl: FC<Omit<P, keyof IntlProps>> = (props) => {
    const intl = useIntl();
    const newProps = {
      ...props,
      intl,
    } as P;

    return <Comp {...newProps} />;
  };
  Intl.displayName = `Intl(${getComponentName(Comp)})`;
  return Intl;
}
