# Latency Tracker

- Provides SSR compatible Benchmark Tracking for React Applications
- Optional Transition Abandonment Reporting

## Installation

```sh
$ yarn add tachyon-latency-tracker
```

This package provides components to handle interactivity tracking for an SSR
Relay/React app. It uses the Performance and UserTiming APIs to generate
timestamps for events relevant to the "time to interactivity" / "customer wait
time" for loading pages.

## Enabling Latency Tracking

Use the `LatencyTrackerRoot` to consume reported latency events in your
application. The following events will be reported by doing so (listed in order
of appearance):

1. [benchmark_fetch_start](https://blueprint.di.xarth.tv/#/schema/benchmark_fetch_start)
1. [benchmark_app_booted](https://blueprint.di.xarth.tv/#/schema/benchmark_app_booted)
1. [benchmark_complete_transition](https://blueprint.di.xarth.tv/#/schema/benchmark_complete_transition)

```tsx
import { LatencyTrackerRoot, LatencyTrackerRootProps } from 'tachyon-latency-tracker';

const appData: LatencyTrackerRootProps['appData'] = { ... };

const root = (
  <LatencyTrackerRoot
    appData={appData}
    currentLocation={...}
    currentPath={window.location.pathname}
    interstitialLocations={[
      // If the app has entry points that serve as non-destinations, like a
      // redirecting location for a PWA, add this optional prop to prevent these
      // locations (based on route/data-science names) from being counted in
      // transition logic
      'app-shell'
    ]}
    onEvent={...}
  >
    <AppComponents />
  </LatencyTrackerRoot/>
);
```

To disable latency tracking in an app without having to alter the component
tree, use the `disableLatencyTracking` prop on `LatencyTrackerRoot`.

## Transition Complete tracking

There are two ways to signal that a route/location has finished rendering its
baseline interactive experience.

### `<LatencyTransitionComplete />`

Render this component when a route has completed rendering. This is meant to be
placed in the opposite branch from a loading UI:

```tsx
if (loading) {
  return <LoadingPage />;
} else {
  return (
    <>
      <LatencyTransitionComplete />
      <ActualPage />
    </>
  );
}
```

This declarative approach is the recommended usage.

### useLatencyTransitionComplete()

This is a hook that can be used more imperatively to signal that a route has
completed rendering.

```tsx
const reportTransitionComplete = useLatencyTransitionComplete();

useEffect(() => {
  if (!loading) {
    reportTransitionComplete();
  }
});
```

This must be run in an effect to ensure that the DOM has been updated _before_
reporting transition complete. This exact structure is used inside
`<LatencyTransitionComplete />`.

### For pages that don't require JS for interactivity

In an SSR application, many pages can be considered interactive before
JavaScript boots if they are the initial page loaded by the user. This means an
app could consider interactivity to happen when `domInteractive` happens in
those situations. Since this requires more attention and decision-making, the
latency framework by default waits until JS boots to report transition complete,
but it is capable of "backdating" the interactivity time to the `domInteractive`
mark. To opt into this behavior, use the `requiresJsForInteractivity` parameter:

```tsx
<LatencyTransitionComplete requiresJsForInteractivity={false} />
```

or

```tsx
useEffect(() => {
  if (!loading) {
    reportTransitionComplete({ requiresJsForInteractivity });
  }
});
```

This only affects the initial page in a navigation session. For subsequent
in-app navigations, normal marking and measuring is used.

### Overriding the reported location

For most use-cases, the current location is passed into the `LatencyTrackerRoot`
at the top of the app. For apps where this is not possible, the transition
complete functionality takes an additional `location` parameter for overriding
the root location value at the call site.

## Visualizing

Using the UserTiming API, the framework generates marks and measures relevant to
the timing of page transitions. These can be seen in the performance monitoring
tab of browser developer tools to get a better sense of how pages are
performing.
