/**
 * The `LatencyEvent` enum lists all the implemented latency events as defined
 * in the
 * [Events names and timing in Benchmark Events Instrumentation Spec (Mobile Web)](https://docs.google.com/document/d/14hzh1w1d3Fyq-fybhguHi9df_IPRByM2T2q9cS30SP0/edit#bookmark=id.6dzww6awanah)
 *
 * Please reference that doc as the source of truth for the definition of these
 * events.
 */
export enum LatencyEventType {
  AppBooted = 'benchmark_app_booted',
  CustomEvent = 'benchmark_custom_event',
  FetchStart = 'benchmark_fetch_start',
  PageInteractive = 'benchmark_complete_transition',
}

/**
 * Base for all latency events.
 */
export type LatencyBaseEvent = {
  /**
   * The ID for this session. This is often the same as the app session ID.
   */
  benchmark_session_id: string;
  /**
   * The current time on the client device.
   */
  // seconds since epoch as float
  client_time: number;
  /**
   * The current page type the user is on (e.g. channel profile page).
   */
  destination: string;
  /**
   * The current page type the user is on (e.g. channel profile page).
   */
  location: string;
};

/**
 * Please refer to
 * [benchmark_fetch_start in Benchmark Events Instrumentation Spec (Mobile Web)](https://docs.google.com/document/d/14hzh1w1d3Fyq-fybhguHi9df_IPRByM2T2q9cS30SP0/edit#bookmark=id.5xzxsy81dur4)
 */
export type LatencyFetchStartEvent = {
  event: LatencyEventType.FetchStart;
};

/**
 * Please refer to
 * [app_booted in Benchmark Events Instrumentation Spec (Mobile Web)](https://docs.google.com/document/d/14hzh1w1d3Fyq-fybhguHi9df_IPRByM2T2q9cS30SP0/edit#bookmark=id.5xzxsy81dur4)
 */
export type LatencyAppBootedEvent = {
  event: LatencyEventType.AppBooted;
};

/**
 * Please refer to
 * [benchmark_complete_transition in Benchmark Events Instrumentation Spec (Mobile Web)](https://docs.google.com/document/d/14hzh1w1d3Fyq-fybhguHi9df_IPRByM2T2q9cS30SP0/edit#bookmark=id.hx51oi7vl80d)
 * [Events names and timing in Benchmark Events Instrumentation Spec (Mobile Web)](https://docs.google.com/document/d/14hzh1w1d3Fyq-fybhguHi9df_IPRByM2T2q9cS30SP0/edit#bookmark=id.6dzww6awanah)
 */
export type LatencyPageInteractiveEvent = {
  event: LatencyEventType.PageInteractive;
  is_app_launch: boolean;
  lost_visibility: boolean;
  // milliseconds, must be int
  time_from_fetch: number;
};

/**
 * Please refer to
 * [Custom Benchmark Events in Benchmark Events Instrumentation Spec (Mobile Web)](https://docs.google.com/document/d/14hzh1w1d3Fyq-fybhguHi9df_IPRByM2T2q9cS30SP0/edit#bookmark=id.xluj0bm8de7g)
 */
export type LatencyCustomEvent = {
  // milliseconds, must be int
  duration: number;
  event: LatencyEventType.CustomEvent;
  group: string;
  is_app_launch: boolean;
  key: string;
  label: string;
  lost_visibility: boolean;
};

export type LatencyEvent =
  | LatencyAppBootedEvent
  | LatencyCustomEvent
  | LatencyFetchStartEvent
  | LatencyPageInteractiveEvent;

export type LatencyFullEvent = LatencyBaseEvent & LatencyEvent;
