import type { FC } from 'react';
import { useMemo, useRef } from 'react';
import { useConst, useEffectOnce } from 'tachyon-utils-react';
import type { LatencyTrackerContext } from '../../context';
import { latencyTrackerContext } from '../../context';
import type { CreateLatencyReporterOpts } from '../../createLatencyReporter';
import {
  createLatencyReporter,
  mockLatencyReporter,
} from '../../createLatencyReporter';
import { isSafeToTrackLatency } from './isSafeToTrackLatency';

export type LatencyTrackerRootProps = Pick<
  CreateLatencyReporterOpts,
  'interstitialLocations' | 'onEvent' | 'sessionID'
> & {
  currentLocation: string;
  currentPath: string;
  /**
   * For disabling the latency tracking system in place
   */
  disableLatencyTracking?: boolean;
};

export const LatencyTrackerRoot: FC<LatencyTrackerRootProps> = ({
  children,
  currentLocation,
  currentPath,
  disableLatencyTracking,
  interstitialLocations,
  onEvent,
  sessionID,
}) => {
  const latencyReporter = useConst(() =>
    isSafeToTrackLatency() && !disableLatencyTracking
      ? createLatencyReporter({
          initialLocation: currentLocation,
          interstitialLocations,
          onEvent,
          sessionID,
        })
      : mockLatencyReporter,
  );
  const currentPathRef = useRef(currentPath);

  useEffectOnce(() => {
    latencyReporter.reportPreAppTimings();
  });

  // We trigger off of path because location can stay the same. For instance,
  // if you navigate from one vod directly to another.
  if (currentPath !== currentPathRef.current) {
    latencyReporter.signalRenderingRoute({
      nextLocation: currentLocation,
    });
    currentPathRef.current = currentPath;
  }

  const ctx: LatencyTrackerContext = useMemo(
    () => ({
      latencyReporter,
    }),
    [latencyReporter],
  );

  return <latencyTrackerContext.Provider children={children} value={ctx} />;
};

LatencyTrackerRoot.displayName = 'LatencyTrackerRoot';
