import { datatype, lorem } from 'faker';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import {
  createLatencyReporter,
  mockLatencyReporter,
} from '../../createLatencyReporter';
import { isSafeToTrackLatency } from './isSafeToTrackLatency';
import { LatencyTrackerRoot } from '.';

jest.mock('./isSafeToTrackLatency', () => ({
  isSafeToTrackLatency: jest.fn(),
}));
const mockisSafeToTrackLatency = isSafeToTrackLatency as jest.Mock;

jest.mock('../../createLatencyReporter', () => ({
  ...jest.requireActual('../../createLatencyReporter'),
  createLatencyReporter: jest.fn(),
}));
const mockCreateLatencyReporter = createLatencyReporter as jest.Mock;

describe(LatencyTrackerRoot, () => {
  const setup = createMountWrapperFactory(LatencyTrackerRoot, () => ({
    currentLocation: lorem.word(),
    currentPath: '/initial/path',
    onEvent: jest.fn(),
    sessionID: datatype.uuid(),
  }));

  beforeEach(() => {
    mockCreateLatencyReporter.mockReset();
    mockisSafeToTrackLatency.mockReset();
  });

  it('uses the mock reporter when latency tracking is not safe', () => {
    setup();
    expect(mockCreateLatencyReporter).not.toHaveBeenCalled();
  });

  it('uses the mock reporter when latency tracking is safe but disableLatencyTracking is true', () => {
    mockisSafeToTrackLatency.mockImplementationOnce(() => true);
    setup({ disableLatencyTracking: true });
    expect(mockCreateLatencyReporter).not.toHaveBeenCalled();
  });

  it('signals rendering route when path changes without location change when latency tracking is safe', () => {
    mockisSafeToTrackLatency.mockImplementationOnce(() => true);
    const mockReportPreAppTimings = jest.fn();
    const mockSignalRenderingRoute = jest.fn();
    mockCreateLatencyReporter.mockImplementationOnce(() => ({
      ...mockLatencyReporter,
      reportPreAppTimings: mockReportPreAppTimings,
      signalRenderingRoute: mockSignalRenderingRoute,
    }));
    const { props, wrapper } = setup();

    expect(mockReportPreAppTimings).toHaveBeenCalledTimes(1);
    expect(mockSignalRenderingRoute).not.toHaveBeenCalled();

    const newPath = '/new/path';

    wrapper.setProps({ currentPath: newPath });

    expect(mockReportPreAppTimings).toHaveBeenCalledTimes(1);
    expect(mockSignalRenderingRoute).toHaveBeenCalledTimes(1);
    expect(mockSignalRenderingRoute).toHaveBeenCalledWith({
      nextLocation: props.currentLocation,
    });

    wrapper.setProps({ currentPath: newPath });
    expect(mockReportPreAppTimings).toHaveBeenCalledTimes(1);
    expect(mockSignalRenderingRoute).toHaveBeenCalledTimes(1);
  });
});
