import type { ComponentType, FC } from 'react';
import { getComponentName } from 'tachyon-utils-react';
import type { ReportCustomLatency } from '../context';
import { useCustomLatency } from '../context';

/**
 * Additional props added to a component when wrapped with the
 * `withCustomLatency` HOC.
 */
export type CustomLatencyProps = {
  /**
   * The custom latency reporter.
   */
  reportLatencyEvent: ReportCustomLatency;
};

/**
 * This HOC wraps a component to enable reporting latency for a custom event.
 * This does so by providing a `reportLatencyEvent` prop that will
 * take a custom event name and report that it is interactive.
 * @param Comp The component that has a custom latency event to report.
 */
// istanbul ignore next: trivial
export function withCustomLatency<P extends CustomLatencyProps>(
  Comp: ComponentType<P>,
): ComponentType<Omit<P, keyof CustomLatencyProps>> {
  const CustomLatencyComponent: FC<Omit<P, keyof CustomLatencyProps>> = (
    props,
  ) => {
    const reportLatencyEvent = useCustomLatency();

    return <Comp {...(props as P)} reportLatencyEvent={reportLatencyEvent} />;
  };

  CustomLatencyComponent.displayName = `CustomLatency(${getComponentName(
    Comp,
  )})`;

  return CustomLatencyComponent;
}
