import { createContext, useCallback, useContext } from 'react';
import type {
  CustomLatencyEvent,
  LatencyReporter,
} from './createLatencyReporter';
import { mockLatencyReporter } from './createLatencyReporter';

export type LatencyTrackerContext = {
  latencyReporter: LatencyReporter;
};

export const latencyTrackerContext = createContext<LatencyTrackerContext>({
  latencyReporter: mockLatencyReporter,
});

/**
 * Used for signalling that a route transition, either initial page load or
 * in-app nav. Returns a function to invoke when your page is considered
 * interactive.
 */
// istanbul ignore next: trivial
export function useLatencyTransitionComplete(): LatencyReporter['signalTransitionComplete'] {
  return useContext(latencyTrackerContext).latencyReporter
    .signalTransitionComplete;
}

export type ReportCustomLatency = (event: CustomLatencyEvent) => () => void;

/**
 * Allows the consumer to report custom latency events for a given route.
 * Each custom event will be reported exactly once. Subsequent reports on the
 * same route for the given event type will result in a no-op.
 */
// istanbul ignore next: trivial
export function useCustomLatency(): ReportCustomLatency {
  const { latencyReporter } = useContext(latencyTrackerContext);

  const reportLatencyEvent = useCallback(
    (event: CustomLatencyEvent) => latencyReporter[event],
    [latencyReporter],
  );

  return reportLatencyEvent;
}
