export type GetPageTimeToInteractiveOpts = {
  pageNumber: number;
  pageTimeOriginMark: string;
  /**
   * For pages that don't require JavaScript for full interactivity, backdates
   * first page interactivity timing to `domInteractive`.
   */
  requiresJsForInteractivity: boolean | undefined;
  /**
   * For explicitly disabling domInteractive backdating. Useful for when the
   * entrypoint was an interstitial location.
   */
  skipBackdate: boolean;
};

/**
 * For reporting page interactivity. Capable of "backdating" to
 * `domInteractive` time for pages that only need HTML to achieve basic
 * interactivity (meaning JS isn't required for scrolling, navigating, etc).
 * Will also wait for `domInteractive` if it hasn't happened yet, as in the case
 * with SWs serving assets "too fast".
 *
 * For visualization purposes, generates performance marks/measures for the
 * UserTiming timeline.
 */
export function getPageTimeToInteractive({
  pageNumber,
  pageTimeOriginMark,
  requiresJsForInteractivity,
  skipBackdate,
}: GetPageTimeToInteractiveOpts): number {
  let endMark = 'domInteractive';
  if (
    skipBackdate ||
    requiresJsForInteractivity ||
    pageNumber > 0 ||
    // we can boot before domInteractive when serving from a service worker
    !window.performance.timing.domInteractive
  ) {
    endMark = `Page${pageNumber}End`;
    window.performance.mark(endMark);
  }

  const measure = `Page${pageNumber}Interactive`;
  window.performance.measure(measure, pageTimeOriginMark, endMark);

  return window.performance.getEntriesByName(measure, 'measure')[0].duration;
}
