import type { LatencyBaseEvent } from '../LatencyEvent';
import { LatencyEventType } from '../LatencyEvent';
import type { GetPageTimeToInteractiveOpts } from './getPageTimeToInteractive';
import { getPageTimeToInteractive } from './getPageTimeToInteractive';
import type { PublicCustomEventReporters } from './getPublicCustomEventReporters';
import { getPublicCustomEventReporters } from './getPublicCustomEventReporters';
import { APP_BOOT_MARK, reportPreAppTimings } from './reportPreAppTimings';
import type {
  CustomEventKeys,
  OnEvent,
  ReportCustomEventOpts,
  ReportStandardEventOpts,
} from './types';

export type { CustomLatencyEvent } from './getPublicCustomEventReporters';

export type CreateLatencyReporterOpts = {
  /**
   * The initial location of the user in the app.
   */
  initialLocation: string;
  /**
   * List of locations that don't correspond to real destinations, such as the
   * TMW app shell path. These locations will be excluded from transition
   * tracking and thus won't reset latency timing, instead waiting for the
   * ensuing page to report interactive.
   */
  interstitialLocations?: string[] | undefined;
  /**
   * A callback to report latency events.
   */
  onEvent: OnEvent;
  /**
   * Data about the application that will be reported with each latency event.
   */
  sessionID: string;
};

type SignalRenderingRouteOpts = {
  nextLocation: string;
};

export type SignalTransitionCompleteOpts = Partial<
  Pick<GetPageTimeToInteractiveOpts, 'requiresJsForInteractivity'>
> & {
  /**
   * @experimental
   * Special-case override for reporting a different location name for the
   * current page.
   */
  location?: string;
};

export type LatencyReporter = PublicCustomEventReporters & {
  /**
   * Reports latency data on the timings involved with getting the app running.
   * Split into separate function to allow deferability since they are all
   * computed from pre-existing marks in the Performance/UserTiming APIs.
   */
  reportPreAppTimings: () => void;
  /**
   * Indicate to the LatencyReporter that the app has transitioned to a new page.
   */
  signalRenderingRoute: (opts: SignalRenderingRouteOpts) => void;
  /**
   * Indicate that the current page has become interactive.
   */
  signalTransitionComplete: (opts?: SignalTransitionCompleteOpts) => void;
};

export function createLatencyReporter({
  initialLocation,
  interstitialLocations = [],
  onEvent,
  sessionID,
}: CreateLatencyReporterOpts): LatencyReporter {
  window.performance.mark(APP_BOOT_MARK);
  const sessionTimeOriginMark = 'navigationStart';
  const sessionTimeOrigin = window.performance.timing[sessionTimeOriginMark];
  const seenEvents: Set<CustomEventKeys> = new Set();

  let pageNumber = 0;
  let lastInteractivePageNumber = -1;
  let skipBackdate = false;
  let pageTimeOriginMark = sessionTimeOriginMark;
  let pageTimeOrigin = sessionTimeOrigin;
  let currentLocation = initialLocation;
  let wasDocumentHidden = false;

  const isFirstPage = () => pageNumber === 0;
  const isInterstitialLocation = () =>
    interstitialLocations.includes(currentLocation);

  const getLatencyBaseEvent = (epochTime: number): LatencyBaseEvent => {
    return {
      benchmark_session_id: sessionID,
      client_time: epochTime / 1000,
      destination: currentLocation,
      location: currentLocation,
    };
  };

  const reportStandardEvent = ({
    data,
    epochTime,
  }: ReportStandardEventOpts) => {
    onEvent({
      ...getLatencyBaseEvent(epochTime),
      ...data,
    });
  };

  const reportCustomEvent = ({
    group,
    key,
    label,
    epochTime = Date.now(),
    duration = epochTime - pageTimeOrigin,
  }: ReportCustomEventOpts): void => {
    if (seenEvents.has(key)) {
      return;
    }

    seenEvents.add(key);
    onEvent({
      ...getLatencyBaseEvent(epochTime),
      duration: Math.round(duration),
      event: LatencyEventType.CustomEvent,
      group,
      is_app_launch: isFirstPage(),
      key,
      label,
      lost_visibility: wasDocumentHidden,
    });
  };

  document.addEventListener('visibilitychange', () => {
    wasDocumentHidden = true;
  });

  return {
    ...getPublicCustomEventReporters(reportCustomEvent),
    reportPreAppTimings: () => {
      reportPreAppTimings({
        reportCustomEvent,
        reportStandardEvent,
        sessionID,
        sessionTimeOrigin,
        sessionTimeOriginMark,
      });
    },
    signalRenderingRoute: ({ nextLocation }) => {
      if (!isInterstitialLocation()) {
        pageNumber++;
        pageTimeOriginMark = `Page${pageNumber}Start`;
        window.performance.mark(pageTimeOriginMark);
        pageTimeOrigin =
          window.performance.getEntriesByName(pageTimeOriginMark, 'mark')[0]
            .startTime + sessionTimeOrigin;
        wasDocumentHidden = false;
        seenEvents.clear();
      }
      currentLocation = nextLocation;
    },
    signalTransitionComplete: ({
      location,
      requiresJsForInteractivity = true,
    } = {}) => {
      if (pageNumber !== lastInteractivePageNumber) {
        if (location) {
          currentLocation = location;
        }
        if (isInterstitialLocation()) {
          skipBackdate = true;
          return;
        }
        lastInteractivePageNumber = pageNumber;
        const timeToInteractive = getPageTimeToInteractive({
          pageNumber,
          pageTimeOriginMark,
          requiresJsForInteractivity,
          skipBackdate,
        });

        reportStandardEvent({
          data: {
            event: LatencyEventType.PageInteractive,
            is_app_launch: isFirstPage(),
            lost_visibility: wasDocumentHidden,
            time_from_fetch: Math.round(timeToInteractive),
          },
          epochTime: pageTimeOrigin + timeToInteractive,
        });
      }
    },
  };
}

export const mockLatencyReporter: LatencyReporter = {
  chatConnected: () => undefined,
  playerPlaying: () => undefined,
  playerReady: () => undefined,
  reportPreAppTimings: () => undefined,
  signalRenderingRoute: () => undefined,
  signalTransitionComplete: () => undefined,
  workerActivated: () => undefined,
  workerControlling: () => undefined,
  workerInstalled: () => undefined,
  workerWaiting: () => undefined,
};
