import { LatencyEventType } from '../LatencyEvent';
import type { ReportCustomEvent, ReportStandardEvent } from './types';
import { CustomEventGroup, CustomEventKeys } from './types';

/**
 * The custom performance entry name to mark when the app has started booting.
 */
export const APP_BOOT_MARK = 'app_boot';
/**
 * The custom performance entry name to mark when the app has finished booting.
 */
const APP_BOOT_MEASURE = 'app_booted';

export type ReportPreAppTimingsOpts = {
  reportCustomEvent: ReportCustomEvent;
  reportStandardEvent: ReportStandardEvent;
  sessionID: string;
  sessionTimeOrigin: number;
  sessionTimeOriginMark: string;
};

// istanbul ignore next: trivial
export function reportPreAppTimings(opts: ReportPreAppTimingsOpts): void {
  reportFetchStart(opts);
  reportConnectEnd(opts);
  reportFirstByte(opts);
  reportAppBooted(opts);

  if (window.document.readyState !== 'complete') {
    const listenerCallback = () => {
      reportFirstCue(opts);
      window.document.removeEventListener('readystatechange', listenerCallback);
    };
    window.document.addEventListener('readystatechange', listenerCallback);
  } else {
    reportFirstCue(opts);
  }
}

/**
 * Reports the fetch start event.
 */
// istanbul ignore next: trivial
function reportFetchStart({
  reportStandardEvent,
  sessionTimeOrigin,
}: ReportPreAppTimingsOpts): void {
  reportStandardEvent({
    data: {
      event: LatencyEventType.FetchStart,
    },
    epochTime: sessionTimeOrigin,
  });
}

/**
 * Reports the establishment of the TCP connection.
 */
// istanbul ignore next: trivial
function reportConnectEnd({
  reportCustomEvent,
}: ReportPreAppTimingsOpts): void {
  reportCustomEvent({
    duration:
      window.performance.timing.connectEnd -
      window.performance.timing.connectStart,
    epochTime: window.performance.timing.connectEnd,
    group: CustomEventGroup.Page,
    key: CustomEventKeys.ConnectEnd,
    label: 'tcp connected',
  });
}

/**
 * Reports the first byte event.
 */
// istanbul ignore next: trivial
function reportFirstByte({ reportCustomEvent }: ReportPreAppTimingsOpts): void {
  reportCustomEvent({
    duration:
      window.performance.timing.responseStart -
      window.performance.timing.requestStart,
    epochTime: window.performance.timing.responseStart,
    group: CustomEventGroup.Page,
    key: CustomEventKeys.FirstByte,
    label: 'first byte',
  });
}

/**
 * Reports the app booted event.
 */
// istanbul ignore next: trivial
function reportAppBooted({
  reportStandardEvent,
  sessionTimeOrigin,
  sessionTimeOriginMark,
}: ReportPreAppTimingsOpts): void {
  window.performance.measure(
    APP_BOOT_MEASURE,
    sessionTimeOriginMark,
    APP_BOOT_MARK,
  );
  const measure = window.performance.getEntriesByName(
    APP_BOOT_MEASURE,
    'measure',
  )[0];
  reportStandardEvent({
    data: {
      event: LatencyEventType.AppBooted,
    },
    epochTime: sessionTimeOrigin + measure.duration,
  });
}

/**
 * Reports the first cue event.
 */
// istanbul ignore next: trivial
function reportFirstCue({
  reportCustomEvent,
  sessionTimeOrigin,
}: ReportPreAppTimingsOpts): void {
  const firstPaintMark = window.performance.getEntriesByName(
    'first-contentful-paint',
  )[0];
  const firstPaintStartTime = firstPaintMark?.startTime;

  const duration = firstPaintStartTime
    ? firstPaintStartTime
    : window.performance.timing.domInteractive - sessionTimeOrigin;

  const epochTime = firstPaintStartTime
    ? sessionTimeOrigin + firstPaintStartTime
    : window.performance.timing.domInteractive;

  reportCustomEvent({
    duration,
    epochTime,
    group: CustomEventGroup.Page,
    key: CustomEventKeys.FirstCue,
    label: firstPaintStartTime ? 'first paint' : 'dom interactive',
  });
}
