import { datatype, lorem } from 'faker';
import { LatencyEventType } from '../LatencyEvent';
import { APP_BOOT_MARK, reportPreAppTimings } from './reportPreAppTimings';
import { CustomEventGroup, CustomEventKeys } from './types';
import { createLatencyReporter } from '.';

jest.mock('./reportPreAppTimings', () => ({
  ...jest.requireActual('./reportPreAppTimings'),
  reportPreAppTimings: jest.fn(),
}));
const mockReportPreAppTimings = reportPreAppTimings as jest.Mock;

describe(createLatencyReporter, () => {
  const sessionTimeOrigin = datatype.number({ min: 1 });

  const mockMark = jest.fn();
  const mockMeasure = jest.fn();
  (window as any).performance = {
    getEntriesByName: () => [
      {
        duration: datatype.number({ min: 1 }),
        startTime: datatype.number({ min: 1 }),
      },
    ],
    mark: mockMark,
    measure: mockMeasure,
    timing: {
      domInteractive: datatype.number({ min: 1 }),
      navigationStart: sessionTimeOrigin,
    },
  };

  beforeEach(() => {
    mockReportPreAppTimings.mockReset();
  });

  const sessionID = datatype.uuid();

  it('automatically marks app boot', () => {
    createLatencyReporter({
      initialLocation: lorem.word(),
      onEvent: () => undefined,
      sessionID,
    });

    expect(mockMark).toHaveBeenLastCalledWith(APP_BOOT_MARK);
  });

  it('delegates to reportPreAppTimings', () => {
    const reporter = createLatencyReporter({
      initialLocation: lorem.word(),
      onEvent: () => undefined,
      sessionID,
    });

    reporter.reportPreAppTimings();
    expect(mockReportPreAppTimings).toHaveBeenLastCalledWith({
      reportCustomEvent: expect.any(Function),
      reportStandardEvent: expect.any(Function),
      sessionID,
      sessionTimeOrigin,
      sessionTimeOriginMark: 'navigationStart',
    });
  });

  it('exposes public custom event reporters without duplication and calls onEvent callback', () => {
    const initialLocation = lorem.word();
    const onEvent = jest.fn();
    const reporter = createLatencyReporter({
      initialLocation,
      onEvent,
      sessionID,
    });
    expect(onEvent).not.toHaveBeenCalled();

    reporter.playerPlaying();
    expect(onEvent).toHaveBeenCalledTimes(1);
    expect(onEvent).toHaveBeenLastCalledWith(
      expect.objectContaining({
        group: CustomEventGroup.Player,
        key: CustomEventKeys.PlayerPlaying,
        label: 'first frame',
        location: initialLocation,
      }),
    );

    reporter.playerPlaying();
    expect(onEvent).toHaveBeenCalledTimes(1);

    reporter.signalRenderingRoute({
      nextLocation: lorem.word(),
    });
    reporter.playerPlaying();
    expect(onEvent).toHaveBeenCalledTimes(2);
  });

  it('handles transition complete reports without duplication', () => {
    const initialLocation = lorem.word();
    const onEvent = jest.fn();
    const reporter = createLatencyReporter({
      initialLocation,
      onEvent,
      sessionID,
    });
    expect(onEvent).not.toHaveBeenCalled();

    reporter.signalTransitionComplete();
    expect(onEvent).toHaveBeenCalledTimes(1);
    expect(onEvent).toHaveBeenLastCalledWith(
      expect.objectContaining({
        event: LatencyEventType.PageInteractive,
        is_app_launch: true,
        location: initialLocation,
      }),
    );

    reporter.signalTransitionComplete();
    expect(onEvent).toHaveBeenCalledTimes(1);

    const nextLocation = lorem.word();
    reporter.signalRenderingRoute({
      nextLocation,
    });
    reporter.signalTransitionComplete();
    expect(onEvent).toHaveBeenCalledTimes(2);
    expect(onEvent).toHaveBeenLastCalledWith(
      expect.objectContaining({
        event: LatencyEventType.PageInteractive,
        is_app_launch: false,
        location: nextLocation,
      }),
    );
  });

  it('handles transition complete reports with overriden location', () => {
    const initialLocation = lorem.word();
    const onEvent = jest.fn();
    const reporter = createLatencyReporter({
      initialLocation,
      onEvent,
      sessionID,
    });
    expect(onEvent).not.toHaveBeenCalled();

    const locationOverride1 = lorem.word();
    reporter.signalTransitionComplete({ location: locationOverride1 });
    expect(onEvent).toHaveBeenCalledTimes(1);
    expect(onEvent).toHaveBeenLastCalledWith(
      expect.objectContaining({
        event: LatencyEventType.PageInteractive,
        is_app_launch: true,
        location: locationOverride1,
      }),
    );

    reporter.signalTransitionComplete();
    expect(onEvent).toHaveBeenCalledTimes(1);

    reporter.signalRenderingRoute({
      nextLocation: lorem.word(),
    });

    const locationOverride2 = lorem.word();
    reporter.signalTransitionComplete({ location: locationOverride2 });
    reporter.signalTransitionComplete();
    expect(onEvent).toHaveBeenCalledTimes(2);
    expect(onEvent).toHaveBeenLastCalledWith(
      expect.objectContaining({
        event: LatencyEventType.PageInteractive,
        is_app_launch: false,
        location: locationOverride2,
      }),
    );
  });

  it('creates correct marks and measures for pages that require JS', () => {
    const reporter = createLatencyReporter({
      initialLocation: lorem.word(),
      onEvent: () => undefined,
      sessionID,
    });
    expect(mockMark).toHaveBeenCalledTimes(1);
    expect(mockMeasure).toHaveBeenCalledTimes(0);

    reporter.signalTransitionComplete();
    expect(mockMark).toHaveBeenCalledTimes(2);
    expect(mockMark).toHaveBeenLastCalledWith('Page0End');
    expect(mockMeasure).toHaveBeenCalledTimes(1);
    expect(mockMeasure).toHaveBeenLastCalledWith(
      'Page0Interactive',
      'navigationStart',
      'Page0End',
    );

    reporter.signalRenderingRoute({
      nextLocation: lorem.word(),
    });
    expect(mockMark).toHaveBeenCalledTimes(3);
    expect(mockMark).toHaveBeenLastCalledWith('Page1Start');
  });

  it('creates correct measures for pages that do not require JS', () => {
    const reporter = createLatencyReporter({
      initialLocation: lorem.word(),
      onEvent: () => undefined,
      sessionID,
    });
    expect(mockMeasure).toHaveBeenCalledTimes(0);

    reporter.signalTransitionComplete({ requiresJsForInteractivity: false });
    expect(mockMeasure).toHaveBeenCalledTimes(1);
    expect(mockMeasure).toHaveBeenLastCalledWith(
      'Page0Interactive',
      'navigationStart',
      'domInteractive',
    );
  });

  it('does not start or complete transitions or backdate for interstitial locations', () => {
    const onEvent = jest.fn();
    const reporter = createLatencyReporter({
      initialLocation: 'interstitial',
      interstitialLocations: ['interstitial'],
      onEvent,
      sessionID,
    });
    expect(mockMark).toHaveBeenCalledTimes(1);
    expect(mockMeasure).toHaveBeenCalledTimes(0);
    expect(onEvent).not.toHaveBeenCalled();

    reporter.signalTransitionComplete();
    expect(onEvent).not.toHaveBeenCalled();
    expect(mockMark).toHaveBeenCalledTimes(1);
    expect(mockMeasure).toHaveBeenCalledTimes(0);

    const nextLocation1 = lorem.word();
    reporter.signalRenderingRoute({
      nextLocation: nextLocation1,
    });
    expect(mockMark).toHaveBeenCalledTimes(1);

    reporter.signalTransitionComplete({ requiresJsForInteractivity: false });
    expect(mockMark).toHaveBeenCalledTimes(2);
    expect(mockMark).toHaveBeenLastCalledWith('Page0End');
    expect(mockMeasure).toHaveBeenCalledTimes(1);
    expect(mockMeasure).toHaveBeenLastCalledWith(
      'Page0Interactive',
      'navigationStart',
      'Page0End',
    );
    expect(onEvent).toHaveBeenCalledTimes(1);
    expect(onEvent).toHaveBeenLastCalledWith(
      expect.objectContaining({
        event: LatencyEventType.PageInteractive,
        is_app_launch: true,
        location: nextLocation1,
      }),
    );

    const nextLocation2 = lorem.word();
    reporter.signalRenderingRoute({
      nextLocation: nextLocation2,
    });
    expect(mockMark).toHaveBeenCalledTimes(3);
    expect(mockMark).toHaveBeenLastCalledWith('Page1Start');

    reporter.signalTransitionComplete();
    expect(mockMark).toHaveBeenCalledTimes(4);
    expect(mockMark).toHaveBeenLastCalledWith('Page1End');
    expect(mockMeasure).toHaveBeenCalledTimes(2);
    expect(mockMeasure).toHaveBeenLastCalledWith(
      'Page1Interactive',
      'Page1Start',
      'Page1End',
    );
    expect(onEvent).toHaveBeenCalledTimes(2);
    expect(onEvent).toHaveBeenLastCalledWith(
      expect.objectContaining({
        event: LatencyEventType.PageInteractive,
        is_app_launch: false,
        location: nextLocation2,
      }),
    );
  });
});
