import type { BrowserOptions } from '@sentry/browser';

const DEFAULT_THROTTLE_THRESHOLD = 0.01;

type BeforeSend = NonNullable<BrowserOptions['beforeSend']>;

export interface BuildBeforeSendOpts {
  /**
   * A callback invoked if an 'unhandled' error is encountered. An 'unhandled' error is
   * an error that would trigger 'window.onerror' or 'window.onunhandledrejection'.
   *
   * Caveats:
   *  - Sentry.Integrations.TryCatch wraps native time and events APIs (setTimeout, setInterval, requestAnimationFrame, addEventListener/removeEventListener) in try/catch blocks to handle async exceptions. Therefore errors in these handlers will _NOT_ trigger the `onUnhandledError` callback.
   *  - See the docs for `Sentry.Integrations.TryCatch` for more context: https://docs.sentry.io/platforms/javascript/#browser-specific
   *
   */
  onUnhandledError?: (() => void) | undefined;
  /**
   * Controls the sampling rate of throttled events sent to Sentry, from 0 to 1
   * with 1 being 100% of events are sent. Note that this value stacks with the
   * overall sampling rate.
   */
  throttledErrorMessageThreshold?: number | undefined;
  /**
   * An array of messages that will be throttled to a lower sampling rate (as
   * opposed to being ignored entirely).
   */
  throttledErrorMessages?: string[] | undefined;
}

export function buildBeforeSend({
  onUnhandledError,
  throttledErrorMessageThreshold = DEFAULT_THROTTLE_THRESHOLD,
  throttledErrorMessages = [],
}: BuildBeforeSendOpts): BeforeSend {
  const beforeSend: BeforeSend = (event, _hint) => {
    // https://github.com/getsentry/sentry-javascript/issues/2249#issuecomment-539959382
    const mechanism = event.exception?.values?.[0]?.mechanism;
    if (mechanism?.handled === false && onUnhandledError) {
      onUnhandledError();
    }

    if (event.message && throttledErrorMessages.includes(event.message)) {
      if (Math.random() > throttledErrorMessageThreshold) {
        return null;
      }
    }

    return event;
  };

  return beforeSend;
}
