import { buildBeforeSend } from '.';

describe(buildBeforeSend, () => {
  it('does not affect unfiltered messages', () => {
    const event = { message: 'message' } as any;
    const beforeSend = buildBeforeSend({});
    expect(beforeSend(event)).toEqual(event);
  });

  describe('throttled messages', () => {
    const throttledErrorMessages = ['throttle me'];
    const event = { message: 'throttle me' } as any;

    it('drops throttled messages above the threshold', () => {
      const beforeSend = buildBeforeSend({
        throttledErrorMessageThreshold: 0,
        throttledErrorMessages,
      });
      expect(beforeSend(event)).toBeNull();
    });

    it('allows throttled messages below the threshold', () => {
      const beforeSend = buildBeforeSend({
        throttledErrorMessageThreshold: 1,
        throttledErrorMessages,
      });
      expect(beforeSend(event)).toEqual(event);
    });
  });

  describe('onUnhandledError', () => {
    it("invokes onUnhandledError for unhandled errors' ", () => {
      const event = {
        exception: { values: [{ mechanism: { handled: false } }] },
      } as any;
      const onUnhandledError = jest.fn();
      const beforeSend = buildBeforeSend({ onUnhandledError });

      beforeSend(event);

      expect(onUnhandledError).toHaveBeenCalledTimes(1);
    });

    it("does not invoke onUnhandledError for handled errors' ", () => {
      const event = {
        exception: { values: [{ mechanism: { handled: true } }] },
      } as any;
      const onUnhandledError = jest.fn();
      const beforeSend = buildBeforeSend({ onUnhandledError });

      beforeSend(event);

      expect(onUnhandledError).not.toHaveBeenCalled();
    });

    it("does not invoke onUnhandledError if handled is not explicitly set to false' ", () => {
      const events: any[] = [
        {},
        { exception: {} },
        { exception: { values: [] } },
        { exception: { values: [{}] } },
        { exception: { values: [{ mechanism: {} }] } },
      ];
      const onUnhandledError = jest.fn();
      const beforeSend = buildBeforeSend({ onUnhandledError });

      events.forEach((event) => {
        beforeSend(event);
      });

      expect(onUnhandledError).not.toHaveBeenCalled();
    });
  });
});
