import type { BrowserOptions } from '@sentry/browser';
import { Integrations, configureScope, init, setUser } from '@sentry/browser';
import type { AppEnvironment } from 'tachyon-environment';
import { stringToSha256 } from 'tachyon-utils-crypto';
import { getDeviceIDInBrowser } from 'tachyon-utils-twitch';
import type { BuildBeforeSendOpts } from './buildBeforeSend';
import { buildBeforeSend } from './buildBeforeSend';

export type ConfigureClientLoggingOpts = BuildBeforeSendOpts &
  Pick<BrowserOptions, 'ignoreErrors'> & {
    /**
     * Environment of the current application instance, sent to Sentry as the
     * "enviornment" value of the app.
     */
    appEnvironment: AppEnvironment;
    /**
     * Sentry DSN for application/project
     */
    appSentryDsn: string;
    /**
     * Build id of the current application instance, sent to Sentry as the
     * "release" value of the app.
     */
    buildId: string;
    /**
     * Language of the current user/browser, sent as a tag on all events.
     */
    language: string;
    /**
     * Platform of the application, sent as a tag on all events.
     */
    platform: string;
    /**
     * Controls the overall sampling rate of events sent to Sentry, from 0 to 1
     * with 1 being 100% of events are sent.
     */
    throttleThreshold?: number;
  };

let sentryWasConfigured = false;

// istanbul ignore next: trivial
export function isSentryConfigured(): boolean {
  return sentryWasConfigured;
}

// istanbul ignore next: high cost low value
export function configureClientLogging({
  appEnvironment,
  appSentryDsn,
  buildId,
  ignoreErrors,
  language,
  onUnhandledError,
  platform,
  throttleThreshold,
  throttledErrorMessageThreshold,
  throttledErrorMessages,
}: ConfigureClientLoggingOpts): void {
  // consider decluttering: https://docs.sentry.io/platforms/javascript/#decluttering-sentry
  init({
    beforeSend: buildBeforeSend({
      onUnhandledError,
      throttledErrorMessageThreshold,
      throttledErrorMessages,
    }),
    defaultIntegrations: false,
    dsn: appSentryDsn,
    enabled: appEnvironment !== 'development',
    environment: appEnvironment,
    // TODO EXACT-TYPES: cast to because ignoreErrors is Optional((string | RegExp[])) instead of Optional((string | RegExp[]) | undefined)
    ignoreErrors: ignoreErrors as string[],
    integrations: [
      // allows developers to ignore specific errors based on the type or message
      new Integrations.InboundFilters(),
      // allows the SDK to provide original functions and method names
      new Integrations.FunctionToString(),
      // wraps native time and events APIs in try/catch blocks to handle async exceptions
      new Integrations.TryCatch(),
      // wraps native APIs to capture breadcrumbs
      new Integrations.Breadcrumbs({
        // Log HTTP requests done with the Beacon API (apparently not yet supported)
        // beacon: true,
        // Log calls to `console.log`, `console.debug`, etc
        console: false,
        // Log all click and keypress events
        dom: true,
        // Log HTTP requests done with the Fetch API
        fetch: true,
        // Log calls to `history.pushState` and friends
        history: true,
        // Log whenever we send an event to the server
        sentry: false,
        // Log HTTP requests done with the XHR API
        xhr: true,
      }),
      // attaches global handlers to capture uncaught exceptions and unhandled rejections
      new Integrations.GlobalHandlers({
        onerror: true,
        onunhandledrejection: true,
      }),
      // allows you to configure linked errors (disabled, values are defaults)
      // new Integrations.LinkedErrors({
      //   key: 'cause',
      //   limit: 5,
      // }),
      // attaches user-agent information to the event
      new Integrations.UserAgent(),
    ],
    release: buildId,
    // TODO EXACT-TYPES: cast to because sampleRate is Optional(number) instead of Optional(number | undefined)
    sampleRate: throttleThreshold as number,
  });

  // we use device id as a proxy for helping Sentry track users
  // salted with platform value
  const sentryId = `${platform}${
    getDeviceIDInBrowser() || Math.random().toString()
  }`;

  setUser({
    // we hash to prevent sending PII to Sentry's servers
    // id is the best match from the available main user attributes
    id: stringToSha256(sentryId),
  });

  configureScope((scope) => {
    scope.setTag('language', language);
    scope.setTag('platform', platform);
  });

  sentryWasConfigured = true;
}
