/* eslint-disable no-console */
import { Severity } from '@sentry/browser';
import { isSentryConfigured } from '../configureClientLogging';
import { recordBreadcrumb } from '../recordBreadcrumb';
import { recordError } from '../recordError';
import type { ErrorLogPayload, LogPayload } from '../types';
import { consoleSerialize } from './consoleSerialize';

const hasConsole = typeof console !== 'undefined';

interface Logger {
  debug: (logPayload: LogPayload) => void;
  error: (logPayload: ErrorLogPayload) => void;
  info: (logPayload: LogPayload) => void;
  log: (logPayload: LogPayload) => void;
  warn: (logPayload: LogPayload) => void;
}

// Not all browsers/contexts support console.debug
// @ts-expect-error: TS thinks console.debug is always defined
const debugMethod = console.debug ? 'debug' : 'log';

export const logger: Logger = {
  debug: (logPayload: LogPayload) => {
    if (hasConsole) {
      console[debugMethod](consoleSerialize(logPayload));
    }

    if (isSentryConfigured()) {
      recordBreadcrumb({ level: Severity.Debug, logPayload });
    }
  },
  error: (errorLogPayload: ErrorLogPayload) => {
    if (hasConsole) {
      // ensure level is present for cloudwatch metric filters
      console.error(
        consoleSerialize({
          ...errorLogPayload,
          level: errorLogPayload.level ?? 'error',
        }),
      );
    }

    if (isSentryConfigured()) {
      recordError(errorLogPayload);
    }
  },
  info: (logPayload: LogPayload) => {
    if (hasConsole) {
      console.info(consoleSerialize(logPayload));
    }

    if (isSentryConfigured()) {
      recordBreadcrumb({ level: Severity.Info, logPayload });
    }
  },
  log: (logPayload: LogPayload) => {
    if (hasConsole) {
      console.log(consoleSerialize(logPayload));
    }

    if (isSentryConfigured()) {
      recordBreadcrumb({ level: Severity.Log, logPayload });
    }
  },
  warn: (logPayload: LogPayload) => {
    if (hasConsole) {
      console.warn(consoleSerialize(logPayload));
    }

    if (isSentryConfigured()) {
      recordBreadcrumb({ level: Severity.Warning, logPayload });
    }
  },
};
