import { Severity } from '@sentry/browser';
import { isSentryConfigured } from '../configureClientLogging';
import { recordBreadcrumb } from '../recordBreadcrumb';
import { recordError } from '../recordError';
import type { ConsoleMethod } from './test-mocks';
import { mockConsole } from './test-mocks';
import { logger } from '.';

jest.mock('../configureClientLogging', () => ({
  isSentryConfigured: jest.fn(() => false),
}));
const mockIsSentryConfigured = isSentryConfigured as jest.Mock;

jest.mock('../recordError', () => ({
  recordError: jest.fn(),
}));
const mockRecordError = recordError as jest.Mock;

jest.mock('../recordBreadcrumb', () => ({
  recordBreadcrumb: jest.fn(),
}));
const mockRecordBreadcrumb = recordBreadcrumb as jest.Mock;

describe('logger', () => {
  const category = 'category';
  const message = 'message';
  const packageName = 'package';
  const logPayload = { category, message, package: packageName };

  interface LogTestcase {
    method: ConsoleMethod;
    severity: Severity;
  }

  const logTestcases: LogTestcase[] = [
    {
      method: 'debug',
      severity: Severity.Debug,
    },
    {
      method: 'info',
      severity: Severity.Info,
    },
    {
      method: 'log',
      severity: Severity.Log,
    },
    {
      method: 'warn',
      severity: Severity.Warning,
    },
  ];

  logTestcases.forEach(({ method, severity }) => {
    describe(method, () => {
      it(`reports to console.${method}`, () => {
        expect(
          mockConsole(method, () => logger[method](logPayload)),
        ).toHaveBeenCalledWith(logPayload);

        expect(mockRecordBreadcrumb).not.toHaveBeenCalled();
        expect(mockRecordError).not.toHaveBeenCalled();
      });

      it('reports to sentry when sentry has been configured', () => {
        mockIsSentryConfigured.mockImplementationOnce(() => true);
        expect(
          mockConsole(method, () => logger[method](logPayload)),
        ).toHaveBeenCalledWith(logPayload);

        expect(mockRecordBreadcrumb).toHaveBeenCalledWith({
          level: severity,
          logPayload,
        });
        expect(mockRecordError).not.toHaveBeenCalled();
      });
    });
  });

  describe('error', () => {
    const level = 'fatal' as const;
    const errorLogPayload = { ...logPayload, level };

    it('reports to console.error', () => {
      expect(
        mockConsole('error', () => logger.error(errorLogPayload)),
      ).toHaveBeenCalledWith(errorLogPayload);

      expect(mockRecordBreadcrumb).not.toHaveBeenCalled();
      expect(mockRecordError).not.toHaveBeenCalled();
    });

    it('reports to console.error with default level when not supplied', () => {
      expect(
        mockConsole('error', () => logger.error(logPayload)),
      ).toHaveBeenCalledWith({ ...logPayload, level: 'error' });

      expect(mockRecordBreadcrumb).not.toHaveBeenCalled();
      expect(mockRecordError).not.toHaveBeenCalled();
    });

    it('reports to sentry when sentry has been configured', () => {
      mockIsSentryConfigured.mockImplementationOnce(() => true);
      expect(
        mockConsole('error', () => logger.error(errorLogPayload)),
      ).toHaveBeenCalledWith(errorLogPayload);

      expect(mockRecordError).toHaveBeenCalledWith(errorLogPayload);
      expect(mockRecordBreadcrumb).not.toHaveBeenCalled();
    });
  });
});
