import {
  Severity,
  captureException,
  captureMessage,
  withScope,
} from '@sentry/browser';
import { recordError } from '.';

jest.mock('@sentry/browser', () => ({
  ...jest.requireActual('@sentry/browser'),
  captureException: jest.fn(),
  captureMessage: jest.fn(),
  withScope: jest.fn(),
}));
const mockCaptureException = captureException as jest.Mock;
const mockCaptureMessage = captureMessage as jest.Mock;
const mockWithScope = withScope as jest.Mock;

describe(recordError, () => {
  const mockSetTag = jest.fn();
  const mockSetContext = jest.fn();
  const mockSetLevel = jest.fn();

  mockWithScope.mockImplementation((cb: any) => {
    const mockScope = {
      setContext: mockSetContext,
      setLevel: mockSetLevel,
      setTag: mockSetTag,
    };

    cb(mockScope);
  });

  const category = 'category';
  const context = { key: 'value' };
  const error = new Error('error');
  const level = 'fatal';
  const message = 'message';
  const packageName = 'package';

  it('processes a fully hydrated error payload', () => {
    recordError({
      category,
      context,
      error,
      level,
      message,
      package: packageName,
    });

    expect(mockSetTag).toHaveBeenCalledTimes(3);
    expect(mockSetTag).toHaveBeenCalledWith('package', packageName);
    expect(mockSetTag).toHaveBeenCalledWith('category', category);
    expect(mockSetTag).toHaveBeenCalledWith('message', message);

    expect(mockSetContext).toHaveBeenCalledTimes(1);
    expect(mockSetContext).toHaveBeenCalledWith('context', context);

    expect(mockSetLevel).toHaveBeenCalledTimes(1);
    expect(mockSetLevel).toHaveBeenCalledWith(Severity.Fatal);

    expect(mockCaptureException).toHaveBeenCalledWith(error);
    expect(mockCaptureMessage).not.toHaveBeenCalled();
  });

  it('processes a minimal error payload', () => {
    recordError({
      category,
      message,
      package: packageName,
    });

    expect(mockSetTag).toHaveBeenCalledTimes(2);
    expect(mockSetTag).toHaveBeenCalledWith('package', packageName);
    expect(mockSetTag).toHaveBeenCalledWith('category', category);

    expect(mockSetContext).not.toHaveBeenCalled();
    expect(mockSetLevel).not.toHaveBeenCalled();

    expect(mockCaptureMessage).toHaveBeenCalledWith(message);
    expect(mockCaptureException).not.toHaveBeenCalled();
  });

  it('does not report when payload has localOnly', () => {
    recordError({
      category,
      localOnly: true,
      message,
      package: packageName,
    });

    expect(mockWithScope).not.toHaveBeenCalled();
    expect(mockCaptureException).not.toHaveBeenCalled();
    expect(mockCaptureMessage).not.toHaveBeenCalled();
  });
});
