import type { ComponentType, ErrorInfo, ReactNode } from 'react';
import { Component } from 'react';
import { logger } from 'tachyon-logger';

export type ClientSideErrorBoundaryProps = {
  app: string;
  boundaryName: string;
  children: ReactNode;
  currentPath?: string;
  errorStateComponent: ComponentType;
  isDevelopment?: boolean;
  isRootBoundary?: true;
  onError?: () => void;
};

export type ClientSideErrorBoundaryState = {
  erroringPath: string | undefined;
  renderErrorState: boolean;
};

/**
 * This is a component for enabling error handling in a render tree via
 * the componentDidCatch lifecycle method. If any component in this component's
 * subtree throws an error during render, this component will stop that error's
 * propagation, render the result of the `errorStateRender` prop, and log the
 * error prepended with `messagePrefix` prop (in order to create error
 * namespaces).
 *
 * You can optionally provide a `currentPath` prop, which is mostly
 * used for root-level error handling so as to allow resetting of this component
 * for trees that persist through in-app navigation. You can also optionally
 * provide an `onError` callback prop, for when you need a parent component to
 * be able to react to errors; this callback provides no information is merely
 * a binary signifier that an error occurred.
 *
 * This component can be used anywhere in the tree, just be sure to provide an
 * `errorStateRender` that appropriately fills the space for the subtree which
 * it is replacing in the UI, as well as a descriptive `messagePrefix` for use
 * in analyzing client-side errors.
 */
export class ClientSideErrorBoundary extends Component<
  ClientSideErrorBoundaryProps,
  ClientSideErrorBoundaryState
> {
  public override state: ClientSideErrorBoundaryState = {
    erroringPath: undefined,
    renderErrorState: false,
  };

  public static displayName = 'ClientSideErrorBoundary';

  public static getDerivedStateFromProps(
    nextProps: ClientSideErrorBoundaryProps,
    prevState: ClientSideErrorBoundaryState,
  ): Partial<ClientSideErrorBoundaryState> | null {
    if (
      prevState.erroringPath &&
      nextProps.currentPath &&
      prevState.erroringPath !== nextProps.currentPath
    ) {
      return {
        erroringPath: undefined,
        renderErrorState: false,
      };
    }

    return null;
  }

  public override componentDidCatch(error: Error, info: ErrorInfo): void {
    // in development mode, re-throw to propagate error up to react-error-overlay
    if (this.props.isDevelopment) {
      throw error;
    }

    if (this.props.onError) {
      this.props.onError();
    }

    this.setState(() => ({
      erroringPath: this.props.currentPath,
      renderErrorState: true,
    }));

    try {
      logger.error({
        category: this.props.boundaryName,
        context: { ...info },
        error,
        level: this.props.isRootBoundary ? 'fatal' : 'error',
        message: error.message ?? 'Unknown boundary error',
        package: this.props.app,
      });
    } catch {
      // Don't let the logger kill us.
    }
  }

  public override render(): ReactNode {
    if (this.state.renderErrorState) {
      return <this.props.errorStateComponent />;
    }

    return this.props.children;
  }
}

export type WithClientSideErrorBoundaryProps = Omit<
  ClientSideErrorBoundaryProps,
  'children' | 'currentPath'
>;
