import type { FC } from 'react';
import { logger } from 'tachyon-logger';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { ClientSideErrorBoundary } from '.';

jest.mock('tachyon-logger', () => ({ logger: { error: jest.fn() } }));

const TestComponent: FC = () => <p>hello</p>;
const ErrorComponent: FC = () => <h1>Error</h1>;

describe(ClientSideErrorBoundary, () => {
  const setup = createShallowWrapperFactory(ClientSideErrorBoundary, () => ({
    app: 'test',
    boundaryName: 'SomeBoundary',
    children: <TestComponent />,
    errorStateComponent: ErrorComponent,
    isDevelopment: false,
  }));

  describe('with no errors', () => {
    it('renders the original content', () => {
      const { wrapper } = setup();
      expect(wrapper.find(TestComponent)).toExist();
      expect(wrapper.find(ErrorComponent)).not.toExist();
      expect(wrapper).toHaveState({
        erroringPath: undefined,
        renderErrorState: false,
      });
    });
  });

  describe('with errors', () => {
    it('re-throws error in development', () => {
      const { wrapper } = setup({ isDevelopment: true });
      expect(() =>
        wrapper.find(TestComponent).simulateError(new Error('test')),
      ).toThrow('test');
    });

    it('logs the error message', () => {
      const { props, wrapper } = setup();
      wrapper.find(TestComponent).simulateError(new Error('test'));

      expect(logger.error).toHaveBeenCalledWith(
        expect.objectContaining({
          category: props.boundaryName,
          level: 'error',
          message: 'test',
          package: props.app,
        }),
      );
    });

    it('logs the error message as fatal for root boundary', () => {
      const { wrapper } = setup({ isRootBoundary: true });
      wrapper.find(TestComponent).simulateError(new Error('test'));

      expect(logger.error).toHaveBeenCalledWith(
        expect.objectContaining({
          level: 'fatal',
        }),
      );
    });

    it('renders the error content', () => {
      const { wrapper } = setup();
      wrapper.find(TestComponent).simulateError(new Error('test'));

      expect(wrapper.find(TestComponent)).not.toExist();
      expect(wrapper.find(ErrorComponent)).toExist();
      expect(wrapper).toHaveState({
        erroringPath: undefined,
        renderErrorState: true,
      });
    });

    it('calls onError when present', () => {
      const onError = jest.fn();
      const { wrapper } = setup({ onError });

      wrapper.find(TestComponent).simulateError(new Error('test'));
      expect(onError).toHaveBeenCalled();
    });

    it('maintains rendering an error when path is set but has not changed across updates', () => {
      const currentPath = '/path1';
      const { wrapper } = setup({ currentPath });

      wrapper.find(TestComponent).simulateError(new Error('test'));
      expect(wrapper.find(ErrorComponent)).toExist();

      wrapper.setProps({ currentPath });
      expect(wrapper).toHaveState({
        erroringPath: currentPath,
        renderErrorState: true,
      });
      expect(wrapper.find(ErrorComponent)).toExist();
    });

    it('resets to rendering children after an error when path changes', () => {
      const originalPath = '/path1';
      const newPath = '/path2';
      const { wrapper } = setup({ currentPath: originalPath });

      wrapper.find(TestComponent).simulateError(new Error('test'));
      expect(wrapper.find(ErrorComponent)).toExist();

      wrapper.setProps({ currentPath: newPath });
      expect(wrapper).toHaveState({
        erroringPath: undefined,
        renderErrorState: false,
      });
      expect(wrapper.find(TestComponent)).toExist();
    });
  });
});
